# -*- coding: utf-8 -*-

# about.py
# This file is part of qarte-5
#    
# Author: Vincent Vande Vyvre <vincent.vandevyvre@oqapy.eu>
# Copyright: 2011-2022 Vincent Vande Vyvre
# Licence: GPL3
# Home page: https://launchpad.net/qarte
#
# About Qarte dialog box

import os
import subprocess
import sys
import platform
import webbrowser

from PyQt5.QtCore import (Qt, QT_VERSION_STR, PYQT_VERSION_STR, QSize, QUrl, 
                         pyqtSignal)
from PyQt5.QtGui import QIcon, QPixmap, QTextCharFormat, QFont, QCursor
from PyQt5.QtWidgets import (QApplication, QDialog, QGridLayout, QVBoxLayout,
                             QHBoxLayout, QSizePolicy, QSpacerItem,
                             QPlainTextEdit, QLabel, QPushButton, QFrame,
                             QTabWidget, QWidget)

from sip import SIP_VERSION_STR

class About:
    def __init__(self, v):
        """Show a dialog box with version infos
        
        Args:
        v -- version of Qarte
        """
        VERSION__STR = v
        PY_VERSION = sys.version.split()[0]
        PLATFORM = platform.platform()
        versions = {"platform": PLATFORM,
                    "Qarte": VERSION__STR,
                    "python": PY_VERSION,
                    "qt": QT_VERSION_STR,
                    "pyqt": PYQT_VERSION_STR,
                    "sip": SIP_VERSION_STR}

        font = QFont("Mono", 10)
        c_form = QTextCharFormat()
        c_form.setFont(font)
        font1 = QFont("sans", 10)
        font1.setWeight(56)
        c_form1 = QTextCharFormat()
        c_form1.setFont(font1)

        ui = UiAbout()
        ui.setupUi(versions)
        ui.show()

        txt = PLATFORM + '\n'
        ui.editor.setCurrentCharFormat(c_form)
        ui.editor.insertPlainText(txt)

        l = 25 - len(VERSION__STR)
        ui.editor.setCurrentCharFormat(c_form1)
        ui.editor.appendPlainText("Qarte")
        txt = "\t" + " " * l + VERSION__STR
        ui.editor.setCurrentCharFormat(c_form)
        ui.editor.insertPlainText(txt)

        l = 25 - len(PY_VERSION)
        ui.editor.setCurrentCharFormat(c_form1)
        ui.editor.appendPlainText("Python")
        txt = "\t" + " " * l + PY_VERSION
        ui.editor.setCurrentCharFormat(c_form)
        ui.editor.insertPlainText(txt)

        l = 25 - len(QT_VERSION_STR)
        ui.editor.setCurrentCharFormat(c_form1)
        ui.editor.appendPlainText("Qt")
        txt = "\t" + " " * l + QT_VERSION_STR
        ui.editor.setCurrentCharFormat(c_form)
        ui.editor.insertPlainText(txt)

        l = 25 - len(PYQT_VERSION_STR)
        ui.editor.setCurrentCharFormat(c_form1)
        ui.editor.appendPlainText("PyQt")
        txt = "\t" + " " * l + PYQT_VERSION_STR
        ui.editor.setCurrentCharFormat(c_form)
        ui.editor.insertPlainText(txt)

        l = 25 - len(SIP_VERSION_STR)
        ui.editor.setCurrentCharFormat(c_form1)
        ui.editor.appendPlainText("Sip")
        txt = "\t" + " " * l + SIP_VERSION_STR
        ui.editor.setCurrentCharFormat(c_form)
        ui.editor.insertPlainText(txt)

        ui.editor_2.insertPlainText(self.get_licence_text())
        #ui.exec_()

    def get_licence_text(self):
        try:
            with open('medias/copyright', 'r') as inf:
                txt = inf.read()
        except:
                txt = '\n\tLicence text not found!'

        return txt

class UiAbout(QDialog):
    def setupUi(self, versions):
        self.versions = versions
        self.setWindowTitle("About Qarte")
        self.resize(543, 455)
        self.gridLayout = QGridLayout(self)
        self.verticalLayout_6 = QVBoxLayout()
        self.horizontalLayout = QHBoxLayout()
        self.verticalLayout = QVBoxLayout()
        spacerItem = QSpacerItem(20, 22, QSizePolicy.Minimum, QSizePolicy.Fixed)
        self.verticalLayout.addItem(spacerItem)
        self.line = QFrame(self)
        sizePolicy = QSizePolicy(QSizePolicy.Expanding, QSizePolicy.Fixed)
        sizePolicy.setHorizontalStretch(0)
        sizePolicy.setVerticalStretch(0)
        sizePolicy.setHeightForWidth(self.line.sizePolicy().hasHeightForWidth())
        self.line.setSizePolicy(sizePolicy)
        self.line.setLineWidth(2)
        self.line.setFrameShape(QFrame.HLine)
        self.line.setFrameShadow(QFrame.Sunken)
        self.verticalLayout.addWidget(self.line)
        self.horizontalLayout.addLayout(self.verticalLayout)
        self.label = QLabel(self)
        self.label.setPixmap(QPixmap("medias/qarte_logo.png"))
        self.horizontalLayout.addWidget(self.label)
        self.verticalLayout_5 = QVBoxLayout()
        spacerItem1 = QSpacerItem(20, 22, QSizePolicy.Minimum, QSizePolicy.Fixed)
        self.verticalLayout_5.addItem(spacerItem1)
        self.line_2 = QFrame(self)
        sizePolicy = QSizePolicy(QSizePolicy.Fixed, QSizePolicy.Fixed)
        sizePolicy.setHorizontalStretch(0)
        sizePolicy.setVerticalStretch(0)
        sizePolicy.setHeightForWidth(self.line_2.sizePolicy().hasHeightForWidth())
        self.line_2.setSizePolicy(sizePolicy)
        self.line_2.setMinimumSize(QSize(40, 0))
        self.line_2.setMaximumSize(QSize(40, 16777215))
        self.line_2.setLineWidth(2)
        self.line_2.setFrameShape(QFrame.HLine)
        self.line_2.setFrameShadow(QFrame.Sunken)
        self.verticalLayout_5.addWidget(self.line_2)
        self.horizontalLayout.addLayout(self.verticalLayout_5)
        self.verticalLayout_6.addLayout(self.horizontalLayout)
        self.horizontalLayout_2 = QHBoxLayout()
        spacerItem2 = QSpacerItem(40, 20, QSizePolicy.Expanding, 
                                        QSizePolicy.Minimum)
        self.horizontalLayout_2.addItem(spacerItem2)
        self.label_2 = QLabel(self)
        self.label_2.setText("Author: Vincent Vande Vyvre")
        self.horizontalLayout_2.addWidget(self.label_2)
        spacerItem3 = QSpacerItem(40, 20, QSizePolicy.Expanding, 
                                        QSizePolicy.Minimum)
        self.horizontalLayout_2.addItem(spacerItem3)
        self.verticalLayout_6.addLayout(self.horizontalLayout_2)
        self.horizontalLayout_3 = QHBoxLayout()
        spacerItem4 = QSpacerItem(40, 20, QSizePolicy.Expanding, 
                                        QSizePolicy.Minimum)
        self.horizontalLayout_3.addItem(spacerItem4)
        self.label_3 = QLabel(self)
        self.label_3.setText("Licence: GPL-3")
        self.horizontalLayout_3.addWidget(self.label_3)
        spacerItem5 = QSpacerItem(40, 20, QSizePolicy.Expanding, 
                                        QSizePolicy.Minimum)
        self.horizontalLayout_3.addItem(spacerItem5)
        self.verticalLayout_6.addLayout(self.horizontalLayout_3)
        self.horizontalLayout_4 = QHBoxLayout()
        spacerItem6 = QSpacerItem(40, 20, QSizePolicy.Expanding, 
                                        QSizePolicy.Minimum)
        self.horizontalLayout_4.addItem(spacerItem6)
        self.label_4 = QLabel(self)
        self.label_4.setText("Sources: ")
        self.horizontalLayout_4.addWidget(self.label_4)
        self.label_5 = UrlLabel("https://launchpad.net/qarte", self)
        self.horizontalLayout_4.addWidget(self.label_5)
        spacerItem7 = QSpacerItem(40, 20, QSizePolicy.Expanding, 
                                        QSizePolicy.Minimum)
        self.horizontalLayout_4.addItem(spacerItem7)
        self.verticalLayout_6.addLayout(self.horizontalLayout_4)
        self.tabWidget = QTabWidget(self)
        self.tab = QWidget()
        self.verticalLayout_3 = QVBoxLayout(self.tab)
        self.verticalLayout_3.setContentsMargins(2, 2, 2, 2)
        self.verticalLayout_2 = QVBoxLayout()
        self.horizontalLayout_5 = QHBoxLayout()
        spacerItem8 = QSpacerItem(288, 20, QSizePolicy.Expanding, 
                                        QSizePolicy.Minimum)
        self.horizontalLayout_5.addItem(spacerItem8)
        self.copy_lbl = QLabel(self)
        font = QFont()
        font.setItalic(True)
        self.copy_lbl.setFont(font)
        self.copy_lbl.setStyleSheet("color: #A0A0A0")
        self.copy_lbl.setText('Copy to the clipboard')
        self.copy_lbl.setCursor(QCursor(Qt.PointingHandCursor))
        self.copy_lbl.mousePressEvent = self.on_copy_clicked
        self.horizontalLayout_5.addWidget(self.copy_lbl)
        self.verticalLayout_2.addLayout(self.horizontalLayout_5)
        self.editor = QPlainTextEdit(self.tab)
        self.editor.setReadOnly(True)
        self.verticalLayout_2.addWidget(self.editor)
        self.verticalLayout_3.addLayout(self.verticalLayout_2)
        self.tabWidget.addTab(self.tab, "Versions")
        self.tab_2 = QWidget()
        self.verticalLayout_4 = QVBoxLayout(self.tab_2)
        self.verticalLayout_4.setContentsMargins(2, 2, 2, 2)
        self.editor_2 = QPlainTextEdit(self.tab_2)
        self.editor_2.setReadOnly(True)
        self.verticalLayout_4.addWidget(self.editor_2)
        self.tabWidget.addTab(self.tab_2, "Licences")
        self.verticalLayout_6.addWidget(self.tabWidget)
        self.gridLayout.addLayout(self.verticalLayout_6, 0, 0, 1, 1)

        self.tabWidget.setCurrentIndex(0)
        self.clip_board = QApplication.clipboard()
        self.tabWidget.currentChanged.connect(self.on_tab_changed)

    def on_tab_changed(self, idx):
        if idx == 1:
            self.editor_2.verticalScrollBar().setValue(0)

    def on_copy_clicked(self, event):
        """Copy all versions nummers into the clipboard.

        The method copy() of QPlainTextEdit don't save the formatting of the
        text, so we re-create it.
        """
        txt = """Qarte\t\t%(Qarte)s\nPlatform\t%(platform)s\nQt\t\t\t%(qt)s
Python\t\t%(python)s\nPyQt\t\t%(pyqt)s\nSip\t\t\t%(sip)s""" % self.versions
        self.clip_board.setText(txt)

class UrlLabel(QLabel):
    def __init__(self, text='', parent=None):
        super(UrlLabel, self).__init__(parent)
        font = QFont()
        font.setUnderline(True)
        self.setFont(font)
        self.setStyleSheet("color: #0000FF")
        self.set_text(text)
        self.setCursor(QCursor(Qt.PointingHandCursor))
        self.clip_board = QApplication.clipboard()

    def set_text(self, text):
        self.url = text
        self.setText(text)

    def mousePressEvent(self, event):
        if event.button() == 1:
            self.open_in_webbroser()

        elif event.button() == 2:
            self.show_context_menu(event.globalPos())

    def show_context_menu(self, pos):
        self.menu = QMenu()
        self.copy = QAction(QIcon('medias/copy.png'), 'Copy', None)
        self.web = QAction(QIcon('medias/web.png'), 'Open', None)
        self.menu.addAction(self.copy)
        self.menu.addAction(self.web)
        self.copy.triggered.connect(self.copy_url)
        self.web.triggered.connect(self.open_in_webbroser)
        self.menu.popup(pos)

    def copy_url(self):
        self.clip_board.setText(self.url)

    def open_in_webbroser(self):
        webbrowser.open_new_tab(self.url)
