/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2016 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#define _BSD_SOURCE /* strdup() */
#define _GNU_SOURCE /* asprintf() */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "cdw_ext_tools.h"
#include "cdw_string.h"
#include "cdw_debug.h"
#include "cdw_widgets.h"
#include "cdw_thread.h"
#include "cdw_logging.h"
#include "cdw_which.h"
#include "cdw_fs.h"
#include "gettext.h"
#include "cdw_task.h"



/**
   \file cdw_ext_tools.c

   \brief File implements code searching for tools such as mkisofs or
   growisofs, storing full paths in file system to these tools,
   implements getter function providing pointer to path to specified
   tool.

   The file also implements code that creates labels for dropdowns, with
   which user can select one of (possibly) few available implementations
   of given tool.

   This file uses term "instance" - its meaning is "executable file of
   tool X, being placed in given directory in file system". Example:
   on my system tool cdrecord has two instances:
   \li /home/acerion/bin/cdrecord
   \li /usr/bin/cdrecord (which is a symbolic link to /usr/bin/wodim)

   Labels mentioned above can be used to create a dropdown widget in
   configuration window, with which user can select one of these two
   instances to perform writing to CD.
*/






/* number of tool sets:
   dvd+rw-tools + cdrtools + libburnia tools = 3
   number of digest tools: md5sum + shaNsum = 6
   6 > 3 */
#define CDW_EXT_TOOLS_N_FAMILIES_MAX 6

/* system default instance of a tool (if it is available in the system at all)
   is listed by "which" as first, and is put on list of available
   implementations as first */
#define CDW_EXT_TOOLS_SYSTEM_DEFAULT_INSTANCE_IND 0



/* UNIX executable names */
#define CDW_TOOL_MKISOFS_NAME            "mkisofs"
#define CDW_TOOL_CDRECORD_NAME           "cdrecord"
#define CDW_TOOL_GROWISOFS_NAME          "growisofs"
#define CDW_TOOL_DVD_RW_FORMAT_NAME      "dvd+rw-format"
#define CDW_TOOL_DVD_RW_MEDIAINFO_NAME   "dvd+rw-mediainfo"
#define CDW_TOOL_XORRISO_NAME            "xorriso"
#define CDW_TOOL_MKUDFFS_NAME            "mkudffs"
#define CDW_TOOL_MD5SUM_NAME             "md5sum"
#define CDW_TOOL_SHA1SUM_NAME            "sha1sum"
#define CDW_TOOL_SHA224SUM_NAME          "sha224sum"
#define CDW_TOOL_SHA256SUM_NAME          "sha256sum"
#define CDW_TOOL_SHA384SUM_NAME          "sha384sum"
#define CDW_TOOL_SHA512SUM_NAME          "sha512sum"

/* These 7 tools below are helpers for creating UDF image
   (a.k.a. mkudffs helpers). They are used exclusively for creating
   UDF image file.

   I don't intend to make it possible to select instances of these
   tools - when invoking them, I just use hardwired names. But I still
   need to verify that they are available on user's system. The
   verification will be performed just like for any other tool: with
   cdw_which. */
#define CDW_TOOL_TRUNCATE_NAME           "truncate"
#define CDW_TOOL_SUDO_NAME               "sudo"
#define CDW_TOOL_MOUNT_NAME              "mount"
#define CDW_TOOL_UMOUNT_NAME             "umount"
#define CDW_TOOL_SYNC_NAME               "sync"
#define CDW_TOOL_RSYNC_NAME              "rsync"




typedef struct {
	/* dropdown labels and IDs for tool dropdowns displayed
	   in config window; user can select a tool implementation
	   when manual configuration is enabled;
	   +1 is for special value "system default" */
	cdw_id_label_t items[CDW_EXT_TOOLS_N_INSTANCES_MAX + 1];
	int n_items;
	/* index of item currently selected from items[];
	   used during manual tool configuration;
	   during automatic configuration cdw always selects first
	   implementation in items[] - system default */
	int current_ind;
} cdw_ext_tools_tool_config_t;




typedef struct {
	/* dropdown labels and IDs for handler dropdowns displayed
	   in config window; user can select a handler when manual
	   configuration is enabled */
	cdw_id_clabel_t items[CDW_EXT_TOOLS_N_FAMILIES_MAX];
	int n_items;
	/* index of item currently selected from items[];
	   used during manual tool configuration */
	int current_ind;
} cdw_ext_tools_handler_config_t;



typedef struct {
	/* name of executable, 20 chars should be enough for everyone */
	char name[20];

	/* is the tool available in system? */
	bool available;
	/* single tool can have multiple instances (implementations) located
	   in different locations in file system;
	   +1 is for special value "system default" */
	char *instances[CDW_EXT_TOOLS_N_INSTANCES_MAX + 1];
	int n_instances;

	/* dropdown index, index of currently selected item in dropdown, and
	   thus index of tool implementation selected by user (can point to
	   "selected by cdw" or "system default" as well); initially set by
	   external tools module to some default value, then can be modified
	   by user in configuration window */
	int current_instance_ind;

} cdw_tool_t;




static struct {
	struct {
		bool manual_selection;

		/* tool for creating stand alone ISO9660 file;
		   mkisofs + xorriso */
		cdw_ext_tools_handler_config_t iso9660_handlers;

		/* Tools for creating stand alone UDF file. */
		cdw_ext_tools_handler_config_t udf_handlers;

		/* cdrecord + xorriso */
		cdw_ext_tools_handler_config_t cd_handlers;

		/* cdrecord + growisofs + xorriso */
		cdw_ext_tools_handler_config_t dvd_handlers;

		cdw_ext_tools_tool_config_t mkisofs;
		cdw_ext_tools_tool_config_t cdrecord;

		/* md5sum, sha1sum, sha244sum, sha256sum, sha384sum, sha512sum */
		cdw_ext_tools_handler_config_t digest_handlers;

	} config;

	cdw_tool_t tools[CDW_EXT_TOOLS_N_TOOLS];
} cdw_ext_tools;




static void cdw_ext_tools_init_labels(void);
static void cdw_ext_tools_print_labels(void);
static void cdw_ext_tools_find_external_tools(void);
static int  cdw_ext_tools_copy_tool_instances(cdw_id_t t, cdw_dll_t *instances);
static void cdw_ext_tools_init_cdw_tools(void);

static void     cdw_ext_tools_build_cd_handlers(void);
static void     cdw_ext_tools_build_dvd_handlers(void);
static void     cdw_ext_tools_build_iso9660_handlers(void);
static void     cdw_ext_tools_build_udf_handlers(void);
static void     cdw_ext_tools_build_digest_handlers(void);
static cdw_rv_t cdw_ext_tools_build_tool_config(cdw_id_t tool_id, cdw_ext_tools_tool_config_t *tool_config);
static void     cdw_ext_tools_build_handlers(cdw_ext_tools_handler_config_t *handlers, cdw_id_clabel_t input_items[], int n_items_max);
// static cdw_rv_t cdw_ext_tools_check_versions(void);

static cdw_id_t cdw_ext_tools_get_manual_disc_handler_family(cdw_disc_simple_type_t simple_type);

static bool cdw_ext_tools_disc_type_supported_by_cdrecord(cdw_disc_type_t disc_type);
static bool cdw_ext_tools_disc_type_supported_by_dvd_rw_tools(cdw_disc_type_t disc_type);
static bool cdw_ext_tools_disc_type_supported_by_xorriso(cdw_disc_type_t disc_type);

static const char *cdw_ext_tools_get_instance_manual(cdw_id_t tool_id);
static const char *cdw_ext_tools_get_instance_auto(cdw_id_t tool_id);
static const char *cdw_ext_tools_get_system_default_instance(cdw_id_t tool_id);
static int         cdw_ext_tools_get_manual_selection_ind(cdw_id_t tool_id);

static CDW_DROPDOWN *cdw_ext_tools_tool_dropdown_new(WINDOW *window, int row, int col, int width, cdw_ext_tools_tool_config_t *tool_config);
static CDW_DROPDOWN *cdw_ext_tools_handler_dropdown_new(WINDOW *window, int row, int col, int width, cdw_ext_tools_handler_config_t *handler_config);

static bool cdw_ext_tools_dvd_rw_tools_available(void);

static bool cdw_ext_tools_set_iso9660_otf_tool_cd(cdw_disc_t *disc, cdw_id_label_t *tool);
static bool cdw_ext_tools_set_iso9660_otf_tool_dvd(cdw_disc_t *disc, cdw_id_label_t *tool);



/* \brief Label in tools dropdown indicating that a tool is not available */
static char label_not_available[20];
static char label_system_default[20];
static char label_no_tool_available[20];
//static char dvd_rw_tools_label[] = "dvd+rw-tools";



static cdw_id_clabel_t cdw_ext_tools_digest_tools[] =
	{{ CDW_TOOL_MD5SUM,      CDW_TOOL_MD5SUM_NAME    },
	 { CDW_TOOL_SHA1SUM,     CDW_TOOL_SHA1SUM_NAME   },
	 { CDW_TOOL_SHA224SUM,   CDW_TOOL_SHA224SUM_NAME },
	 { CDW_TOOL_SHA256SUM,   CDW_TOOL_SHA256SUM_NAME },
	 { CDW_TOOL_SHA384SUM,   CDW_TOOL_SHA384SUM_NAME },
	 { CDW_TOOL_SHA512SUM,   CDW_TOOL_SHA512SUM_NAME }};





/**
   \brief Initialize "external tools" module

   The function runs "which" utility for each "tool" (such as cdrecord
   or md5sum) to check whether a tool is available in user's system.
   Collected information is put into cdw_tools[].

   \return CDW_OK
*/
cdw_rv_t cdw_ext_tools_init(void)
{
	cdw_ext_tools_init_labels();
	cdw_ext_tools_init_cdw_tools();
	cdw_ext_tools_find_external_tools();
	// cdw_ext_tools_check_versions();
	cdw_ext_tools_build_cd_handlers();
	cdw_ext_tools_build_dvd_handlers();
	cdw_ext_tools_build_iso9660_handlers();
	cdw_ext_tools_build_udf_handlers();
	cdw_ext_tools_build_digest_handlers();

	cdw_ext_tools_build_tool_config(CDW_TOOL_MKISOFS, &cdw_ext_tools.config.mkisofs);
	cdw_ext_tools_build_tool_config(CDW_TOOL_CDRECORD, &cdw_ext_tools.config.cdrecord);

#ifndef NDEBUG  /* only for debug purposes */
	cdw_ext_tools_debug_print_config();
	cdw_ext_tools_print_labels();
#endif
	return CDW_OK;
}





/**
   \brief Assign very initial values to fields in cdw_tools[] variable

   Call this function to initialize fields in cdw_tools[]. This is a first
   step to have correct and useful values in cdw_tools[]. After calling
   this function you are ready to call cdw_ext_tools_find_external_tools().
*/
void cdw_ext_tools_init_cdw_tools(void)
{
	for (int t = 0; t < CDW_EXT_TOOLS_N_TOOLS; t++) {

		/* +1 is for special value "system default" */
		for (int i = 0; i < CDW_EXT_TOOLS_N_INSTANCES_MAX + 1; i++) {
			cdw_ext_tools.tools[t].instances[i] = (char *) NULL;
		}

		cdw_ext_tools.tools[t].name[0] = '\0';
		cdw_ext_tools.tools[t].available = false;
		cdw_ext_tools.tools[t].n_instances = 0;
		cdw_ext_tools.tools[t].current_instance_ind = 0;
	}

	/* it is cdw logic that (by default) selects tools for performing
	   tasks, including selecting proper kind of tool for given task
	   (cdrecord / growisofs / xorriso) if more than one is available */
	cdw_ext_tools.config.manual_selection = false;

	cdw_ext_tools.config.dvd_handlers.current_ind = 0;
	cdw_ext_tools.config.dvd_handlers.n_items = 0;
	cdw_ext_tools.config.cd_handlers.current_ind = 0;
	cdw_ext_tools.config.cd_handlers.n_items = 0;
	cdw_ext_tools.config.iso9660_handlers.current_ind = 0;
	cdw_ext_tools.config.iso9660_handlers.n_items = 0;
	cdw_ext_tools.config.udf_handlers.current_ind = 0;
	cdw_ext_tools.config.udf_handlers.n_items = 0;
	cdw_ext_tools.config.digest_handlers.current_ind = 0;
	cdw_ext_tools.config.digest_handlers.n_items = 0;

	for (int t = 0; t < CDW_EXT_TOOLS_N_FAMILIES_MAX; t++) {
		cdw_ext_tools.config.dvd_handlers.items[t].id = CDW_TOOL_NONE;
		cdw_ext_tools.config.dvd_handlers.items[t].label = (char *) NULL;
		cdw_ext_tools.config.cd_handlers.items[t].id = CDW_TOOL_NONE;
		cdw_ext_tools.config.cd_handlers.items[t].label = (char *) NULL;
		cdw_ext_tools.config.iso9660_handlers.items[t].id = CDW_TOOL_NONE;
		cdw_ext_tools.config.iso9660_handlers.items[t].label = (char *) NULL;
		cdw_ext_tools.config.udf_handlers.items[t].id = CDW_TOOL_NONE;
		cdw_ext_tools.config.udf_handlers.items[t].label = (char *) NULL;
		cdw_ext_tools.config.digest_handlers.items[t].id = CDW_TOOL_NONE;
		cdw_ext_tools.config.digest_handlers.items[t].label = (char *) NULL;
	}

	cdw_ext_tools.config.mkisofs.current_ind = 0;
	cdw_ext_tools.config.mkisofs.n_items = 0;
	cdw_ext_tools.config.cdrecord.current_ind = 0;
	cdw_ext_tools.config.cdrecord.n_items = 0;

	for (int t = 0; t < CDW_EXT_TOOLS_N_INSTANCES_MAX + 1; t++) {
		cdw_ext_tools.config.cdrecord.items[t].id = CDW_TOOL_NONE;
		cdw_ext_tools.config.cdrecord.items[t].label = (char *) NULL;
		cdw_ext_tools.config.mkisofs.items[t].id = CDW_TOOL_NONE;
		cdw_ext_tools.config.mkisofs.items[t].label = (char *) NULL;
	}

	strcpy(cdw_ext_tools.tools[CDW_TOOL_MKISOFS].name,   CDW_TOOL_MKISOFS_NAME);
	strcpy(cdw_ext_tools.tools[CDW_TOOL_CDRECORD].name,  CDW_TOOL_CDRECORD_NAME);
	strcpy(cdw_ext_tools.tools[CDW_TOOL_GROWISOFS].name, CDW_TOOL_GROWISOFS_NAME);
	strcpy(cdw_ext_tools.tools[CDW_TOOL_DVD_RW_FORMAT].name,    CDW_TOOL_DVD_RW_FORMAT_NAME);
	strcpy(cdw_ext_tools.tools[CDW_TOOL_DVD_RW_MEDIAINFO].name, CDW_TOOL_DVD_RW_MEDIAINFO_NAME);
	strcpy(cdw_ext_tools.tools[CDW_TOOL_XORRISO].name,   CDW_TOOL_XORRISO_NAME);
	strcpy(cdw_ext_tools.tools[CDW_TOOL_MKUDFFS].name,   CDW_TOOL_MKUDFFS_NAME);

	strcpy(cdw_ext_tools.tools[CDW_TOOL_MD5SUM].name,    CDW_TOOL_MD5SUM_NAME);
	strcpy(cdw_ext_tools.tools[CDW_TOOL_SHA1SUM].name,   CDW_TOOL_SHA1SUM_NAME);
	strcpy(cdw_ext_tools.tools[CDW_TOOL_SHA224SUM].name, CDW_TOOL_SHA224SUM_NAME);
	strcpy(cdw_ext_tools.tools[CDW_TOOL_SHA256SUM].name, CDW_TOOL_SHA256SUM_NAME);
	strcpy(cdw_ext_tools.tools[CDW_TOOL_SHA384SUM].name, CDW_TOOL_SHA384SUM_NAME);
	strcpy(cdw_ext_tools.tools[CDW_TOOL_SHA512SUM].name, CDW_TOOL_SHA512SUM_NAME);

	strcpy(cdw_ext_tools.tools[CDW_TOOL_TRUNCATE].name, CDW_TOOL_TRUNCATE_NAME);
	strcpy(cdw_ext_tools.tools[CDW_TOOL_SUDO].name,     CDW_TOOL_SUDO_NAME);
	strcpy(cdw_ext_tools.tools[CDW_TOOL_MOUNT].name,    CDW_TOOL_MOUNT_NAME);
	strcpy(cdw_ext_tools.tools[CDW_TOOL_UMOUNT].name,   CDW_TOOL_UMOUNT_NAME);
	strcpy(cdw_ext_tools.tools[CDW_TOOL_SYNC].name,     CDW_TOOL_SYNC_NAME);
	strcpy(cdw_ext_tools.tools[CDW_TOOL_RSYNC].name,    CDW_TOOL_RSYNC_NAME);

	return;
}





/**
   \brief Call "which"-like function with names of external tools

   The function calls cdw_which() for every tool in cdw_tools[] table
   to collect full paths to all instances of every tool. Then the function
   checks potential target files of links if the full paths are paths of
   symbolic links.
*/
void cdw_ext_tools_find_external_tools(void)
{
	for (int t = 0; t < CDW_EXT_TOOLS_N_TOOLS; t++) {
		/* instances = list with paths to every executable named
		   tools[t].name, that is available in paths specified in
		   environment variable PATH */
		cdw_dll_t *instances = (cdw_dll_t *) NULL;
		cdw_vdm ("INFO: looking for tool #%d (%s)\n", t, cdw_ext_tools.tools[t].name);
		if (!cdw_which(cdw_ext_tools.tools[t].name, &instances)) {
			cdw_vdm ("WARNING: can't get instances table for tool \"%s\"\n",
				 cdw_ext_tools.tools[t].name);
		}

		if (t == CDW_TOOL_MKISOFS) {
			cdw_which("genisoimage", &instances);
		} else if (t == CDW_TOOL_CDRECORD) {
			cdw_which("wodim", &instances);
		} else {
			;
		}

		if (!instances) {
			/* no paths for given tool */
			continue;
		}

		cdw_ext_tools.tools[t].n_instances = cdw_ext_tools_copy_tool_instances(t, instances);

		cdw_assert (cdw_ext_tools.tools[t].n_instances <= CDW_EXT_TOOLS_N_INSTANCES_MAX,
			    "ERROR: too many tool instances: %d, limit is %d\n",
			    cdw_ext_tools.tools[t].n_instances, CDW_EXT_TOOLS_N_INSTANCES_MAX);

		/* paths to instances are strdup()ed by copy_tool_instances(),
		   so instances are no longer needed */
		cdw_dll_delete(&instances, free);

		if (cdw_ext_tools.tools[t].n_instances == 0) {
			cdw_ext_tools.tools[t].available = false;
			cdw_ext_tools.tools[t].current_instance_ind = 0;
		} else {
			cdw_ext_tools.tools[t].available = true;
			/* this sets "system default" as default dropdown value */
			cdw_ext_tools.tools[t].current_instance_ind = cdw_ext_tools.tools[t].n_instances;
		}
	}

	return;
}





/**
   \brief Copy given tool fullpaths to cdw_ext_tools.tools[t].

   The function should be used to make copies of tool fullpaths
   returned by cdw_which (table of instances), and put the copies
   into cdw_ext_tools.tools[t].instances table.

   The function does not check if number of instances is less than
   CDW_EXT_TOOLS_N_INSTANCES_MAX. It is assumed that the limit is
   not exceeded.

   \param t - tool id for which \p instances were collected
   \param instances - list of fullpaths to tool \p t

   \return number of instances copied from \p instances
*/
int cdw_ext_tools_copy_tool_instances(cdw_id_t t, cdw_dll_t *instances)
{
	int i = 0;
	for (; instances; instances = instances->next) {
		cdw_ext_tools.tools[t].instances[i] = strdup(instances->data);
		if (!cdw_ext_tools.tools[t].instances[i]) {
			cdw_vdm ("ERROR: failed to strdup item #%d (%s)\n", i, (char *) instances->data);
		} else {
			cdw_vdm ("INFO: fullpath to executable = \"%s\"\n",
				 cdw_ext_tools.tools[t].instances[i]);
		}
		i++;
	}

	return i;
}





/**
   \brief Check assertions about variable describing default paths, print current ext tools settings

   User can use configuration window and its dropdown to select
   default instance of any available tool. Configuration code modifies
   cdw_tools[<tool id>].index, and goal of this function is to
   check (only check) if cdw_tools[<tool id>].index has correct
   value (value within certain range).

   Since it uses assert()s, it is only useful in debug builds.
*/
void cdw_ext_tools_debug_print_config(void)
{
	cdw_vdm ("INFO: %s selection\n\n", cdw_ext_tools.config.manual_selection ? "manual" : "automatic");

	int i = cdw_ext_tools.config.iso9660_handlers.current_ind;
	cdw_assert (cdw_ext_tools.config.iso9660_handlers.items[i].label != (char *) NULL, "ERROR: label #%d of ISO9660 handler is NULL\n", i);
	cdw_vdm ("INFO: ISO9660 #%d label = \"%s\"\n", i, cdw_ext_tools.config.iso9660_handlers.items[i].label);


	/* UDF handlers aren't mutually exclusive alternatives, they
	   are a set of tools needed by "create UDF file" recipe. So
	   don't list "current" tool, because "current" doesn't make
	   sense in this context. List all tools from the recipe. */
	for (int j = 0;
	     j < cdw_ext_tools.config.udf_handlers.n_items;
	     j++) {

		/* Some tools needed by the recipe may be
		   missing. Don't assert. */
#if 0
		cdw_assert (cdw_ext_tools.config.udf_handlers.items[j].label != (char *) NULL, "ERROR: label #%d of UDF handler is NULL\n", j);
#endif
		cdw_vdm ("INFO: UDF tool #%d label = \"%s\"\n", j, cdw_ext_tools.config.udf_handlers.items[j].label);
	}


	i = cdw_ext_tools.config.cd_handlers.current_ind;
	cdw_assert (cdw_ext_tools.config.cd_handlers.items[i].label != (char *) NULL, "ERROR: label #%d of CD handler is NULL\n", i);
	cdw_vdm ("INFO: CD handler #%d label = \"%s\"\n", i, cdw_ext_tools.config.cd_handlers.items[i].label);

	i = cdw_ext_tools.config.dvd_handlers.current_ind;
	cdw_assert (cdw_ext_tools.config.dvd_handlers.items[i].label != (char *) NULL, "ERROR: label #%d of DVD handler is NULL\n", i);
	cdw_vdm ("INFO: DVD handler #%d label = \"%s\"\n", i, cdw_ext_tools.config.dvd_handlers.items[i].label);

	for (int t = 0; t < CDW_EXT_TOOLS_N_TOOLS; t++) {
		int ind = cdw_ext_tools.tools[t].current_instance_ind;
		char *name = cdw_ext_tools.tools[t].name;

		if (cdw_ext_tools.tools[t].available) {
			cdw_vdm ("INFO: fullpaths for %s tool:\n", cdw_ext_tools.tools[t].name);

			for (int j = 0; j < cdw_ext_tools.tools[t].n_instances; j++) {
				cdw_assert (cdw_ext_tools.tools[t].instances[j] != (char *) NULL,
					    "ERROR: which: fullpath %d is null\n", i);

				cdw_vdm ("INFO: fullpath %d = \"%s\"\n", j, cdw_ext_tools.tools[t].instances[j]);
			}

			if (cdw_ext_tools.tools[t].current_instance_ind < cdw_ext_tools.tools[t].n_instances) {
				cdw_vdm ("INFO: path to selected instance is \"%s\"\n",
					cdw_ext_tools.tools[t].instances[cdw_ext_tools.tools[t].current_instance_ind]);
			} else if (cdw_ext_tools.tools[t].current_instance_ind == cdw_ext_tools.tools[t].n_instances) {
				cdw_vdm ("INFO: path to selected instance is \"system default\"\n");
			} else {
				cdw_assert (0, "ERROR: cdw_ext_tools.tools[%d].current_item_index = %d is out of range\n",
					    t, cdw_ext_tools.tools[t].current_instance_ind);
			}
		} else {
			cdw_vdm ("INFO: tool #%d / %s is not available\n", t, cdw_ext_tools.tools[t].name);
			/* tool is not available, so following conditions must be true: */
			cdw_assert (ind == 0,
				    "ERROR: tool %s is marked as not available, but value of its dropdown index is non-zero: %d\n",
				    name, ind);
			cdw_assert (cdw_ext_tools.tools[t].instances[ind] == (char *) NULL,
				    "ERROR: tool %s is marked as not available, but fullpath %d of the tool is not null: \"%s\"\n",
				    name, ind, cdw_ext_tools.tools[t].instances[ind]);

		}
	}

	cdw_vdm ("(newline)\n");

	return;
}





/**
   \brief Deallocate all resources allocated by "external tools" module

   The function deallocates all resources (labels, other strings and other
   memory) used by this module. You should call this function when closing
   application.
*/
void cdw_ext_tools_clean(void)
{
	for (cdw_id_t t = 0; t < CDW_EXT_TOOLS_N_TOOLS; t++) {
		for (int i = 0; i < cdw_ext_tools.tools[t].n_instances; i++) {
			if (cdw_ext_tools.tools[t].instances[i] != (char *) NULL) {
				free(cdw_ext_tools.tools[t].instances[i]);
				cdw_ext_tools.tools[t].instances[i] = (char *) NULL;
			}
		}
	}

	for (int i = 0; i < cdw_ext_tools.config.mkisofs.n_items; i++) {
		if (cdw_ext_tools.config.mkisofs.items[i].label == label_not_available
		     || cdw_ext_tools.config.mkisofs.items[i].label == label_system_default) {
			; /* label allocated by gettext system, must not be free()d */
		} else {
			free(cdw_ext_tools.config.mkisofs.items[i].label);
			cdw_ext_tools.config.mkisofs.items[i].label = (char *) NULL;
		}
	}
	for (int i = 0; i < cdw_ext_tools.config.cdrecord.n_items; i++) {
		if (cdw_ext_tools.config.cdrecord.items[i].label == label_not_available
		    || cdw_ext_tools.config.cdrecord.items[i].label == label_system_default) {
			; /* label allocated by gettext system, must not be free()d */
		} else {
			free(cdw_ext_tools.config.cdrecord.items[i].label);
			cdw_ext_tools.config.cdrecord.items[i].label = (char *) NULL;
		}
	}

	return;
}





/**
   \brief Create id/label tables used later for tool configuration dropdowns

   Function populates \p tool_config table with IDs and labels that can
   be used later to create tool configuration dropdown.
   IDs in such table are simply indexes of items in the table.
   Labels in such table are something more meaningful.

   Labels in \p tool_config table are fullpaths to instances of a given tool.
   If a fullpath is only a symbolic link, the target of the link is appended
   to the fullpath.

   Thus a list of IDs/labels in resulting \p tool_config may look like this:
   tool_config->items[0].label = "/home/acerion/bin/cdrecord"
   tool_config->items[0].id = 0;
   tool_config->items[1].label =  = "/usr/bin/cdrecord -> wodim"
   tool_config->items[1].id = 1;

   If a given tool is not available, then the only label is sth like
   "not available".

   \return CDW_OK
*/
cdw_rv_t cdw_ext_tools_build_tool_config(cdw_id_t tool_id, cdw_ext_tools_tool_config_t *tool_config)
{
	if (!cdw_ext_tools.tools[tool_id].available) {
		tool_config->items[0].label = label_not_available;
		tool_config->items[0].id = CDW_TOOL_NONE;
		tool_config->n_items = 1; /* one for "not available" label */
		tool_config->current_ind = 0; /* "not available" item */
		return CDW_OK;
	}

	cdw_assert (cdw_ext_tools.tools[tool_id].instances[0] != (char *) NULL,
		    "ERROR: tool %s is available, but its first fullpath = NULL\n",
		    cdw_ext_tools.tools[tool_id].name);

	for (int i = 0; i < cdw_ext_tools.tools[tool_id].n_instances; i++) {

		char *fp = cdw_ext_tools.tools[tool_id].instances[i];
		cdw_assert (fp != (char *) NULL, "ERROR: fullpath to instance #%d of tool %lld is NULL\n", i, tool_id);

		char *target = cdw_fs_get_target_path_simple(fp);
		if (target == (char *) NULL) { /* given full path is not a symlink */
			tool_config->items[i].label = strdup(fp);
		} else {
			tool_config->items[i].label = cdw_string_concat(fp, " -> ", target, (char *) NULL);
			free(target);
			target = (char *) NULL;
		}
		tool_config->items[i].id = i;
	}

	int n = cdw_ext_tools.tools[tool_id].n_instances;
        tool_config->items[n].label = label_system_default;
	tool_config->n_items = cdw_ext_tools.tools[tool_id].n_instances + 1; /* +1 is for "system default" label */
	tool_config->current_ind = cdw_ext_tools.tools[tool_id].current_instance_ind; /* previous selection, may be "system default" */

	return CDW_OK;
}





/**
   \brief Print values of variables in cdw_tools[] which will be used as dropdown labels

   Function walks through all tools in cdw_tools[] and prints to stderr all
   "labels" fields available. These fields can be used to create dropdowns
   in configuration window.

   Purpose of this function is purely for debugging.
*/
void cdw_ext_tools_print_labels(void)
{
	for (int t = 0; t < CDW_EXT_TOOLS_N_TOOLS; t++) {
		for (int i = 0; i < cdw_ext_tools.tools[t].n_instances; i++) {
			cdw_assert (cdw_ext_tools.tools[t].instances[i],
				    "ERROR: fullpath to instance #%d of tool %d is NULL\n", i, t);
			fprintf(stderr, " + %s_labels[%d] = \"%s\"\n",
				cdw_ext_tools.tools[t].name, i,
				cdw_ext_tools.tools[t].instances[i]);
		}
	}

	return;
}





/**
   \brief Check if given tool is available (accessible)

   Check if cdw found given tool (specified by \p tool_id).

   \param tool_id - id of a tool (as defined in header file)

   \return true if given tool is available
   \return false if given tool is not available
*/
bool cdw_ext_tools_is_tool_available(cdw_id_t tool_id)
{
	cdw_assert (tool_id >= CDW_TOOL_MKISOFS && tool_id <= CDW_TOOL_RSYNC,
		    "ERROR: incorrect tool id: %lld\n", tool_id);

	return cdw_ext_tools.tools[tool_id].available;
}





/**
   \brief Message informing that a tool can't be found

   Use the function to inform user that cdw can't find given tool (specified
   by \p tool_id), and because of this cdw can't perform required operation.

   If all goes well then tool name (indicated by \p tool_id) is used in
   dialog message, so that user knows which tool causes problems.
   However if something goes wrong, a message without the tool name is
   displayed - user doesn't know which tool causes problems, but still knows
   that there is a problem. In the first case CDW_OK is returned, in second
   case CDW_GEN_ERROR is returned.

   \param tool_id - id of a tool (as defined in header file)

   \return CDW_OK on success
   \return CDW_GEN_ERROR on problems
*/
cdw_rv_t cdw_ext_tools_error_dialog(cdw_id_t tool_id)
{
	cdw_assert (tool_id >= CDW_TOOL_MKISOFS && tool_id <= CDW_TOOL_SHA512SUM,
		    "ERROR: incorrect tool id: %lld\n", tool_id);

	/* 2TRANS: this is message printed into cdw log. %s is a tool name,
	   like cdrecord or growisofs; "Tools" is a label of one of tabs
	   in Configuration window; keep escaped quotes */
	cdw_logging_write(_("ERROR: cdw can't find following tool: \"%s\".\n"), cdw_ext_tools.tools[tool_id].name);

	char *msg = (char *) NULL;
	/* 2TRANS: this is message in dialog window; %s is a tool name,
	   like cdrecord or growisofs; "Tools" is a label of one of tabs
	   in Configuration window; keep escaped quotes */
	int rv = asprintf(&msg, _("cdw can't proceed because it can't find this tool: %s. Please check your options (\"Tools\" tab)."),
			  cdw_ext_tools.tools[tool_id].name);

	if (rv == -1) {
		cdw_vdm ("ERROR: asprintf() returns -1\n");
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window;
				      "tool" refers to external programs like cdrecord
				      or growisofs; keep escaped quotes */
				   _("Can't find some external tool. Please check your options (\"Tools\" tab)."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		return CDW_ERROR;
	} else {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"), msg, CDW_BUTTONS_OK, CDW_COLORS_ERROR);

		free(msg);
		msg = (char *) NULL;
		return CDW_OK;
	}
}





void cdw_ext_tools_init_labels(void)
{
	/* 2TRANS: given tool is not available in user's environment;
	   no more than 19 chars */
	strncpy(label_not_available, _("(not available)"), 19);
	label_not_available[19] = '\0';

	/* 2TRANS: "system default" means "tool selected by default
	   in user's system";  no more than 19 chars */
	strncpy(label_system_default, _("(system default)"), 19);
	label_system_default[19] = '\0';

	/* 2TRANS: "no tool available" means "no tool that would meet some
	   specification is available"; no more than 19 chars */
	strncpy(label_no_tool_available, _("(no tool available)"), 19);
	label_no_tool_available[19] = '\0';

	return;
}





/**
   Copy id/label items from (possibly sparse) \p input_items to
   non-sparse \p handlers, set handlers->n_items. Don't put in \p handlers
   tools that aren't available in user's system.
*/
void cdw_ext_tools_build_handlers(cdw_ext_tools_handler_config_t *handlers, cdw_id_clabel_t input_items[], int n_items_max)
{
	int n_items = 0;
	for (int i = 0; i < n_items_max; i++) {
		if (input_items[i].id != CDW_TOOL_NONE && cdw_ext_tools_is_tool_available(input_items[i].id)) {
			handlers->items[n_items].id = input_items[i].id;
			handlers->items[n_items].label = input_items[i].label;
			n_items++;
		}
	}

	if (n_items == 0) {
		/* there are no handlers available */
		handlers->items[0].label = label_no_tool_available;
		handlers->items[0].id = CDW_TOOL_NONE;
		n_items = 1;  /* the only item will be "not available" */
	}

	/* in all cases first label in dropdown is preselected by default */
	handlers->current_ind = 0;
	handlers->n_items = n_items;

#ifndef NDEBUG
	for (int i = 0; i < handlers->n_items; i++) {
		cdw_assert (handlers->items[i].label != (char *) NULL,
			    "ERROR: label #%d of handlers is NULL\n", i);
		cdw_vdm ("INFO: handler #%d label = \"%s\"\n",
			 i, handlers->items[i].label);
	}
#endif

	return;
}





/* cdrecord alone is enough to get meta info from CD disc,
   write to CD disc and erase CD disc, so no need to check
   availability of other members of cdrtools;
   if mkisofs is also available then inform about it using
   proper label */
const char *clabel_cdrecord = "cdrecord";
const char *clabel_cdrecord_mkisofs  = "cdrecord (+ mkisofs)";

void cdw_ext_tools_build_cd_handlers(void)
{
	/* current algorithm for selecting the best default handler for CDs
	   is simple: cdrtools is always the best;
	   order of checking handlers availability for purposes
	   of creating "CD handlers" dropdown follows the same algorithm:
	   cdrtools is always the best, so check cdrtools
	   first; if the handler exist then put it on top of dropdown */
	bool mkisofs = cdw_ext_tools_is_tool_available(CDW_TOOL_MKISOFS);
	cdw_id_clabel_t tools[] = {
		{ CDW_TOOL_CDRECORD,  mkisofs ? clabel_cdrecord_mkisofs : clabel_cdrecord },
		{ CDW_TOOL_XORRISO,   "xorriso" }};
	int n_items = 2;

	cdw_ext_tools_build_handlers(&(cdw_ext_tools.config.cd_handlers), tools, n_items);

	return;
}





void cdw_ext_tools_build_digest_handlers(void)
{
	cdw_ext_tools_build_handlers(&(cdw_ext_tools.config.digest_handlers), cdw_ext_tools_digest_tools, CDW_EXT_TOOLS_N_DIGEST_TOOLS);
	return;
}





void cdw_ext_tools_build_iso9660_handlers(void)
{
	/* current algorithm for selecting the best default tool for
	   creating stand alone ISO9660 file is simple: mkisofs is
	   always the best;
	   order of checking handlers availability for purposes
	   of creating "iso9660" dropdown follows the same algorithm:
	   mkisofs is always the best, so check mkisofs first; if it
	   exist then put it on top of dropdown */
	cdw_id_clabel_t tool_ids[] = {{ CDW_TOOL_MKISOFS, CDW_TOOL_MKISOFS_NAME },
				      { CDW_TOOL_XORRISO, CDW_TOOL_XORRISO_NAME }};
	cdw_ext_tools_build_handlers(&(cdw_ext_tools.config.iso9660_handlers), tool_ids, 2);

	return;
}





void cdw_ext_tools_build_udf_handlers(void)
{
	/* For now the list includes only a tool that directly handles
	   UDF file system, although a full recipe for creating UDF
	   file perhaps includes few more tools.

	   Perhaps the tools should be counted as "UDF handlers" as well.

	   For now I don't intend to support any UDF creators other
	   than mkudffs, so I won't have to include them in the list
	   below. So the list may be "a list of all tools needed in
	   all steps of recipe for creating UDF file" - udf_recipe_tools. */

	/* For now let's just put here the mkudffs, I will add rest of
	   tools later. */

	cdw_id_clabel_t tool_ids[] = {{ CDW_TOOL_MKUDFFS, CDW_TOOL_MKUDFFS_NAME }};
	cdw_ext_tools_build_handlers(&(cdw_ext_tools.config.udf_handlers), tool_ids, 1);

	return;
}





const char *clabel_dvd_rw_tools_mkisofs = "dvd+rw-tools (+ mkisofs)";
const char *clabel_dvd_rw_tools = "dvd+rw-tools";

void cdw_ext_tools_build_dvd_handlers(void)
{
	/* current algorithm for selecting the best DVD handler is simple:
	   dvd+rw-tools is always the best;
	   order of checking tool sets availability for purposes
	   of creating "DVD handlers" dropdown follows the same algorithm:
	   dvd+rw-tools is always the best, so check dvd+rw-tools
	   first; if the handler exist then put it on top of dropdown */
	bool mkisofs = cdw_ext_tools_is_tool_available(CDW_TOOL_MKISOFS);
	cdw_id_clabel_t tools[] = {{ CDW_TOOL_GROWISOFS, mkisofs ? clabel_dvd_rw_tools_mkisofs : clabel_dvd_rw_tools },
				   { CDW_TOOL_CDRECORD,  mkisofs ? clabel_cdrecord_mkisofs : clabel_cdrecord },
				   /* 2TRANS: this is label in dropdown list; it allows selecting xorriso,
				      with information that you can handle "only" DVD+R and DVD-R discs with it */
				   { CDW_TOOL_XORRISO,   _("xorriso (only DVD+/-R)")     }};
	int n_items = 3;

	if (!cdw_ext_tools_dvd_rw_tools_available()) {
		/* dvd+rw-tools package will be skipped when building dvd handlers */
		tools[0].id = CDW_TOOL_NONE;
	}

	cdw_ext_tools_build_handlers(&(cdw_ext_tools.config.dvd_handlers), tools, n_items);

	return;
}





bool cdw_ext_tools_dvd_rw_tools_available(void)
{
	if (cdw_ext_tools_is_tool_available(CDW_TOOL_GROWISOFS)
	    && cdw_ext_tools_is_tool_available(CDW_TOOL_DVD_RW_MEDIAINFO)
	    && cdw_ext_tools_is_tool_available(CDW_TOOL_DVD_RW_FORMAT)) {

		return true;
	} else {
		return false;
	}
}





bool cdw_ext_tools_iso9660_sa_tool_available(void)
{
	if (cdw_ext_tools_is_tool_available(CDW_TOOL_MKISOFS)
	    || cdw_ext_tools_is_tool_available(CDW_TOOL_XORRISO)) {

		return true;
	} else {
		return false;
	}
}





bool cdw_ext_tools_udf_sa_tools_tools_available(void)
{
	if (cdw_ext_tools_is_tool_available(CDW_TOOL_MKUDFFS)
	    && cdw_ext_tools_is_tool_available(CDW_TOOL_TRUNCATE)
	    && cdw_ext_tools_is_tool_available(CDW_TOOL_SUDO)
	    && cdw_ext_tools_is_tool_available(CDW_TOOL_MOUNT)
	    && cdw_ext_tools_is_tool_available(CDW_TOOL_UMOUNT)
	    && cdw_ext_tools_is_tool_available(CDW_TOOL_SYNC)
	    && cdw_ext_tools_is_tool_available(CDW_TOOL_RSYNC)) {

		return true;
	} else {
		return false;
	}
}





bool cdw_ext_tools_config_is_manual(void)
{
	return cdw_ext_tools.config.manual_selection;
}





const char *cdw_ext_tools_get_tool_name(cdw_id_t tool_id)
{
	/* name of a tool is always defined, even if no implementation
	   of this tool is available */
	return cdw_ext_tools.tools[tool_id].name;
}





bool cdw_ext_tools_is_cdrecord_wodim(void)
{
	/* 1: get current fullpath to cdrecord */
	const char *fullpath = cdw_ext_tools_get_tool_fullpath(CDW_TOOL_CDRECORD);

	/* 2: check if the fullpath itself points to wodim */
	char *c = strstr(fullpath, "wodim");
	if (c != (char *) NULL) {
		cdw_vdm ("INFO: current cdrecord instance IS wodim\n");
		return true;
	}

	/* 3: check if the fullpath is a symlink to wodim */
	char *target = cdw_fs_get_target_path_simple(fullpath);
	if (target == (char *) NULL) { /* given full path is not a symlink */
		cdw_vdm ("INFO: current cdrecord instance IS NOT wodim\n");
		return false;
	} else {
		c = strstr(target, "wodim");
		free(target);
		target = (char *) NULL;
		if (c != (char *) NULL) {
			cdw_vdm ("INFO: current cdrecord instance IS wodim\n");
			return true;
		} else {
			cdw_vdm ("INFO: current cdrecord instance IS NOT wodim\n");
			return false;
		}
	}
}





const char *cdw_ext_tools_get_tool_fullpath(cdw_id_t tool_id)
{
	if (cdw_ext_tools.config.manual_selection) {
		cdw_vdm ("INFO: selecting tool: manual selection\n");
		return cdw_ext_tools_get_instance_manual(tool_id);
	} else {
		cdw_vdm ("INFO: selecting tool: automatic selection\n");
		return cdw_ext_tools_get_instance_auto(tool_id);
	}
}





/**
   \brief Get instance (fullpath) of a tool when "manual configuration" is in effect

   \param tool_id - tool for which to get a fullpath
*/
const char *cdw_ext_tools_get_instance_manual(cdw_id_t tool_id)
{
	char *name = cdw_ext_tools.tools[tool_id].name;

	if (cdw_ext_tools.tools[tool_id].available) {
		/* select implementation */
		int ind = cdw_ext_tools.tools[tool_id].current_instance_ind;
		if (ind == cdw_ext_tools.tools[tool_id].n_instances) {
			/* user selected "system default" in dropdown */
			const char *fullpath = cdw_ext_tools_get_system_default_instance(tool_id);
			cdw_vdm ("INFO: selecting tool: sys default select \"%s\" of tool \"%s\"\n",
				 fullpath, name);
			return fullpath;
		} else {
			const char *fullpath = cdw_ext_tools.tools[tool_id].instances[ind];
			cdw_vdm ("INFO: selecting tool: manual select #%d = \"%s\" of tool \"%s\"\n",
				 ind, fullpath, name);
			return fullpath;
		}
	} else {
		cdw_vdm ("ERROR: tool \"%s\" is unavailable\n", cdw_ext_tools.tools[tool_id].name);
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window;
				      "tool" means "cdrecord", "growisofs", etc. */
				   _("Can't find any tool for current task, please check your configuration and system.\n"),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);

		return (char *) NULL;
	}
}





const char *cdw_ext_tools_get_instance_auto(cdw_id_t tool_id)
{
	const char *fullpath = cdw_ext_tools_get_system_default_instance(tool_id);
	cdw_vdm ("INFO: selecting tool: auto select \"%s\" of tool %s\n",
		 fullpath, cdw_ext_tools.tools[tool_id].name);
	return fullpath;
}





const char *cdw_ext_tools_get_system_default_instance(cdw_id_t tool_id)
{
	return cdw_ext_tools.tools[tool_id].instances[CDW_EXT_TOOLS_SYSTEM_DEFAULT_INSTANCE_IND];
}





CDW_DROPDOWN *cdw_ext_tools_make_dvd_handlers_dropdown(WINDOW *window, int row, int col, int width)
{
	return cdw_ext_tools_handler_dropdown_new(window, row, col, width, &cdw_ext_tools.config.dvd_handlers);
}
CDW_DROPDOWN *cdw_ext_tools_make_cd_handlers_dropdown(WINDOW *window, int row, int col, int width)
{
	return cdw_ext_tools_handler_dropdown_new(window, row, col, width, &cdw_ext_tools.config.cd_handlers);
}
CDW_DROPDOWN *cdw_ext_tools_make_iso9660_dropdown(WINDOW *window, int row, int col, int width)
{
	return cdw_ext_tools_handler_dropdown_new(window, row, col, width, &cdw_ext_tools.config.iso9660_handlers);
}
CDW_DROPDOWN *cdw_ext_tools_make_mkisofs_dropdown(WINDOW *window, int row, int col, int width)
{
	return cdw_ext_tools_tool_dropdown_new(window, row, col, width, &cdw_ext_tools.config.mkisofs);
}
CDW_DROPDOWN *cdw_ext_tools_make_cdrecord_dropdown(WINDOW *window, int row, int col, int width)
{
	return cdw_ext_tools_tool_dropdown_new(window, row, col, width, &cdw_ext_tools.config.cdrecord);
}
CDW_DROPDOWN *cdw_ext_tools_make_digest_handlers_dropdown(WINDOW *window, int row, int col, int width)
{
	return cdw_ext_tools_handler_dropdown_new(window, row, col, width, &cdw_ext_tools.config.digest_handlers);
}





CDW_DROPDOWN *cdw_ext_tools_tool_dropdown_new(WINDOW *window, int row, int col, int width, cdw_ext_tools_tool_config_t *tool_config)
{
	CDW_DROPDOWN *dd = cdw_dropdown_new(window, row, col, width, tool_config->n_items, CDW_COLORS_DIALOG);
	if (dd == (CDW_DROPDOWN *) NULL) {
		cdw_vdm ("ERROR: failed to create new dropdown\n");
		return (CDW_DROPDOWN *) NULL;
	}
	for (int i = 0; i < tool_config->n_items; i++) {
		cdw_rv_t crv = cdw_dropdown_add_item(dd,
						     tool_config->items[i].id,
						     tool_config->items[i].label);
		if (crv != CDW_OK) {
			cdw_vdm ("ERROR: failed to add item %d / \"%s\"",
				 i, tool_config->items[i].label);
			cdw_dropdown_delete(&dd);
			return (CDW_DROPDOWN *) NULL;
		}
	}
	cdw_rv_t crv = cdw_dropdown_finalize(dd);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to finalize dropdown\n");
		cdw_dropdown_delete(&dd);
		return (CDW_DROPDOWN *) NULL;
	}
	crv = cdw_dropdown_set_current_item_by_ind(dd, tool_config->current_ind);
	cdw_assert (crv == CDW_OK, "ERROR: failed to set current dropdown index\n");

	return dd;
}





CDW_DROPDOWN *cdw_ext_tools_handler_dropdown_new(WINDOW *window, int row, int col, int width, cdw_ext_tools_handler_config_t *handler_config)
{
	CDW_DROPDOWN *dd = cdw_dropdown_new(window, row, col, width,
					    handler_config->n_items,
					    CDW_COLORS_DIALOG);
	if (dd == (CDW_DROPDOWN *) NULL) {
		cdw_vdm ("ERROR: failed to create new dropdown\n");
		return (CDW_DROPDOWN *) NULL;
	}
	for (int i = 0; i < handler_config->n_items; i++) {
		cdw_rv_t crv = cdw_dropdown_add_item(dd,
						     handler_config->items[i].id,
						     handler_config->items[i].label);
		if (crv != CDW_OK) {
			cdw_dropdown_delete(&dd);
			cdw_vdm ("ERROR: failed to add item %d / \"%s\"\n", i, handler_config->items[i].label);
			return (CDW_DROPDOWN *) NULL;
		}
	}
	cdw_rv_t crv = cdw_dropdown_finalize(dd);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to finalize dropdown\n");
		cdw_dropdown_delete(&dd);
		return (CDW_DROPDOWN *) NULL;
	}
	crv = cdw_dropdown_set_current_item_by_ind(dd, handler_config->current_ind);
	cdw_assert (crv == CDW_OK, "ERROR: failed to set current dropdown index\n");

	return dd;
}





void cdw_ext_tools_save_configuration(cdw_ext_tools_configuration_t *etc)
{
	cdw_ext_tools.config.manual_selection = etc->manual_selection;

	if (cdw_ext_tools.config.manual_selection) {
		cdw_ext_tools.config.cd_handlers.current_ind = cdw_dropdown_get_current_item_ind(etc->cd_handlers);
		cdw_ext_tools.config.dvd_handlers.current_ind = cdw_dropdown_get_current_item_ind(etc->dvd_handlers);
		cdw_ext_tools.config.iso9660_handlers.current_ind = cdw_dropdown_get_current_item_ind(etc->iso9660);

		/* Is this necessary? The list is not presented in UI,
		   so it won't be changed by user. */
		/* I'm commenting this out because valgrind complains
		   about using uninitialized variable.  etc->udf is
		   never set, is never changed by user in UI, so I
		   think I shouldn't use it.  udf_handlers.current_ind
		   is set only once (elsewhere) and never changed, so
		   this line is not necessary. */
		/* cdw_ext_tools.config.udf_handlers.current_ind = cdw_dropdown_get_current_item_ind(etc->udf); */

		cdw_ext_tools.config.digest_handlers.current_ind = cdw_dropdown_get_current_item_ind(etc->digest_handlers);

		cdw_ext_tools.config.mkisofs.current_ind = cdw_dropdown_get_current_item_ind(etc->mkisofs);
		cdw_ext_tools.config.cdrecord.current_ind = cdw_dropdown_get_current_item_ind(etc->cdrecord);

		cdw_ext_tools.tools[CDW_TOOL_MKISOFS].current_instance_ind = cdw_dropdown_get_current_item_ind(etc->mkisofs);
		cdw_ext_tools.tools[CDW_TOOL_CDRECORD].current_instance_ind = cdw_dropdown_get_current_item_ind(etc->cdrecord);
	}

	return;
}





/* currently only cdrecord and mkisofs have separate dropdowns in
   which you can select an instance (implementation of a tool);
   in all other cases the selected instance is system default instance;

   the code should be probably updated to reflect this */
int cdw_ext_tools_get_manual_selection_ind(cdw_id_t tool_id)
{
	int ind = cdw_ext_tools.tools[tool_id].current_instance_ind;
	if (ind == cdw_ext_tools.tools[tool_id].n_instances) {
		/* user selected "system default" in dropdown, ind = 0 */
		return CDW_EXT_TOOLS_SYSTEM_DEFAULT_INSTANCE_IND;
	} else {
		return ind; /* 0 <= ind < n_instances */
	}
}





bool cdw_ext_tools_disc_type_supported_by_xorriso(cdw_disc_type_t disc_type)
{
	if (disc_type == CDW_CD_R
	    || disc_type == CDW_CD_RW
	    || disc_type == CDW_DVD_R
	    || disc_type == CDW_DVD_R_SEQ
	    || disc_type == CDW_DVD_R_RES
	    || disc_type == CDW_DVD_RP) {

		return true;
	} else {
		return false;
	}
}





bool cdw_ext_tools_disc_type_supported_by_dvd_rw_tools(cdw_disc_type_t disc_type)
{
	if (disc_type == CDW_DVD_R
	    || disc_type == CDW_DVD_R_SEQ
	    || disc_type == CDW_DVD_R_RES
	    || disc_type == CDW_DVD_RP
	    || disc_type == CDW_DVD_RW
	    || disc_type == CDW_DVD_RW_SEQ
	    || disc_type == CDW_DVD_RW_RES
	    || disc_type == CDW_DVD_RWP
	    || disc_type == CDW_DVD_RP_DL) {

		return true;
	} else {
		return false;
	}
}





bool cdw_ext_tools_disc_type_supported_by_cdrecord(__attribute__((unused)) cdw_disc_type_t disc_type)
{
	return true;
}





cdw_id_t cdw_ext_tools_get_manual_disc_handler_family(cdw_disc_simple_type_t simple_type)
{
	if (simple_type == CDW_DISC_SIMPLE_TYPE_CD) {
		int ind = cdw_ext_tools.config.cd_handlers.current_ind;
		cdw_id_t id = cdw_ext_tools.config.cd_handlers.items[ind].id;
		cdw_vdm ("INFO: selecting manual CD handler id = %lld\n", id);
		return id;
	} else if (simple_type == CDW_DISC_SIMPLE_TYPE_DVD) {
		int ind = cdw_ext_tools.config.dvd_handlers.current_ind;
		cdw_id_t id = cdw_ext_tools.config.dvd_handlers.items[ind].id;
		cdw_vdm ("INFO: selecting manual DVD handler id = %lld\n", id);
		return id;
	} else {
		cdw_vdm ("ERROR: disc is neither CD nor DVD\n");
		return CDW_TOOL_NONE;
	}
}





bool cdw_ext_tools_set_burn_tool(cdw_disc_t *disc, cdw_id_label_t *burn_tool, cdw_id_label_t *iso9660_tool)
{
	/* either the function was called for burning image file to
	   disc (and thus iso9660_tool->id == NONE), or burning
	   tool is available and is MKISOFS */
	bool mkisofs_iso_ok = iso9660_tool->id == CDW_TOOL_NONE || iso9660_tool->id == CDW_TOOL_MKISOFS;
	bool xorriso_iso_ok = iso9660_tool->id == CDW_TOOL_NONE || iso9660_tool->id == CDW_TOOL_XORRISO;

	int impl_ind = 0;
	if (cdw_ext_tools.config.manual_selection) {
		cdw_id_t id = cdw_ext_tools_get_manual_disc_handler_family(disc->cdio->simple_type);
		if (id == CDW_TOOL_CDRECORD && mkisofs_iso_ok) {
			burn_tool->id = CDW_TOOL_CDRECORD;
		} else if (id == CDW_TOOL_GROWISOFS && mkisofs_iso_ok) {
			burn_tool->id = CDW_TOOL_GROWISOFS;
		} else if (id == CDW_TOOL_XORRISO && xorriso_iso_ok) {
			burn_tool->id = CDW_TOOL_XORRISO;
		} else {
			cdw_vdm ("ERROR: returning false 1\n");
			return false;
		}

		if (burn_tool->id == CDW_TOOL_XORRISO) {
			if (!cdw_ext_tools_disc_type_supported_by_xorriso(disc->type)) {
				/* 2TRANS: this is title of dialog window */
				cdw_buttons_dialog(_("Information"),
						   /* 2TRANS: this is message in dialog window */
						   _("cdw can't handle this type of disc using xorriso, please select different tool in Configuration -> Tools."),
						   CDW_BUTTONS_OK, CDW_COLORS_DIALOG);
				cdw_vdm ("ERROR: returning false 2\n");
				return false;
			}
		}
		impl_ind = cdw_ext_tools_get_manual_selection_ind(burn_tool->id);
	} else {
		if (cdw_ext_tools_disc_type_supported_by_dvd_rw_tools(disc->type)
		    && cdw_ext_tools_is_tool_available(CDW_TOOL_GROWISOFS)
		    && mkisofs_iso_ok) {

			burn_tool->id = CDW_TOOL_GROWISOFS;

		} else if (cdw_ext_tools_disc_type_supported_by_cdrecord(disc->type)
			   && cdw_ext_tools_is_tool_available(CDW_TOOL_CDRECORD)
			   && mkisofs_iso_ok) {

			burn_tool->id = CDW_TOOL_CDRECORD;

		} else if (cdw_ext_tools_disc_type_supported_by_xorriso(disc->type)
			   && cdw_ext_tools_is_tool_available(CDW_TOOL_XORRISO)
			   && xorriso_iso_ok) {

			burn_tool->id = CDW_TOOL_XORRISO;
		} else {
			cdw_vdm ("ERROR: returning false 3\n");
			return false;
		}
		impl_ind = CDW_EXT_TOOLS_SYSTEM_DEFAULT_INSTANCE_IND;
	}

	burn_tool->label = cdw_ext_tools.tools[burn_tool->id].instances[impl_ind];
	if (burn_tool->label == (char *) NULL) {
		cdw_vdm ("ERROR: burn tool label is NULL (tool id = %lld, impl ind = %d)\n", burn_tool->id, impl_ind);
		cdw_vdm ("ERROR: returning false 4\n");
		return false;
	} else {
		return true;
	}
}





bool cdw_ext_tools_set_digest_tool(cdw_id_label_t *tool)
{
	tool->id = CDW_TOOL_NONE;
	int impl_ind = CDW_EXT_TOOLS_SYSTEM_DEFAULT_INSTANCE_IND;

	if (cdw_ext_tools.config.manual_selection) {
		int ind = cdw_ext_tools.config.digest_handlers.current_ind;
		cdw_id_t id = cdw_ext_tools.config.digest_handlers.items[ind].id;
		if (id != CDW_TOOL_NONE) {
			if (cdw_ext_tools_is_tool_available(id)) {
				tool->id = id;
			}
		}
	} else {
		for (int i = 0; i < CDW_EXT_TOOLS_N_DIGEST_TOOLS; i++) {
			if (cdw_ext_tools_is_tool_available(cdw_ext_tools_digest_tools[i].id)) {
				tool->id = cdw_ext_tools_digest_tools[i].id;
				break;
			}
		}
	}

	if (tool->id == CDW_TOOL_NONE) {
		return false;
	}
	tool->label = cdw_ext_tools.tools[tool->id].instances[impl_ind];
	if (tool->label == (char *) NULL) {
		cdw_vdm ("ERROR: fullpath #%d to digest tool %lld is NULL\n", impl_ind, tool->id);
		return false;
	} else {
		return true;
	}
}





bool cdw_ext_tools_set_iso9660_otf_tool(cdw_disc_t *disc, cdw_id_label_t *tool)
{
	int impl_ind = 0;
	if (cdw_ext_tools.config.manual_selection) {

		/* check which burn handler is selected in configuration
		   (burn handler for CD or DVD), and then pick matching
		   ISO9660 creator;

		   be careful when selecting MKISOFS: the fact that
		   selected burn handler is cdrecord or growisofs doesn't
		   mean that mkisofs is available; do necessary check for
		   availability of mkisofs before picking it */

		cdw_id_t id = cdw_ext_tools_get_manual_disc_handler_family(disc->cdio->simple_type);
		if ((id == CDW_TOOL_CDRECORD || id == CDW_TOOL_GROWISOFS)
		    && cdw_ext_tools_is_tool_available(CDW_TOOL_MKISOFS)) {
			tool->id = CDW_TOOL_MKISOFS;
		} else if (id == CDW_TOOL_XORRISO
			   && cdw_ext_tools_disc_type_supported_by_xorriso(disc->type)) {
			tool->id = CDW_TOOL_XORRISO;
		} else {
			return false;
		}

		impl_ind = cdw_ext_tools_get_manual_selection_ind(tool->id);
	} else {
		/* automatic selection of ISO9660 creator; algorithm:
		   select mkisofs if it exists, and if cdrecord|growisofs exists too;
		   select xorriso otherwise;
		   return false if above fails;

		   use system default implementation of selected ISO9660 creator */

		if (disc->cdio->simple_type == CDW_DISC_SIMPLE_TYPE_CD) {
			bool have_tool = cdw_ext_tools_set_iso9660_otf_tool_cd(disc, tool);
			if (!have_tool) {
				return false;
			}

		} else if (disc->cdio->simple_type == CDW_DISC_SIMPLE_TYPE_DVD) {
			bool have_tool = cdw_ext_tools_set_iso9660_otf_tool_dvd(disc, tool);
			if (!have_tool) {
				return false;
			}
		} else {
			return false;
		}
		impl_ind = CDW_EXT_TOOLS_SYSTEM_DEFAULT_INSTANCE_IND;
	}

	tool->label = cdw_ext_tools.tools[tool->id].instances[impl_ind];
	if (tool->label == (char *) NULL) {
		cdw_vdm ("ERROR: fullpath #%d to ISO9660 tool %lld is NULL\n", impl_ind, tool->id);
		return false;
	} else {
		return true;
	}
}





/**
   \brief Select ISO9660 fs creator suitable for given CD disc

   \date Function's top-level comment reviewed on 2012-02-11
   \date Function's body reviewed on 2012-02-11

   Examine type and state of a CD \p disc and select appropriate
   tool for creating ISO9660 file system on-the-fly (otf).

   The function should be called only when an automatic selection
   of a tool should be performed, because it uses some logic and
   guessing to get the tool.

   Function sets 'id' field of \p tool.

   \param disc - disc to which to examine
   \param tool - variable, in which to set 'id' field

   \return true if tool id has been set successfully
   \return false if tool id has not been set successfully
*/
bool cdw_ext_tools_set_iso9660_otf_tool_cd(cdw_disc_t *disc, cdw_id_label_t *tool)
{
	tool->id = CDW_TOOL_NONE;

	/* first check if there is already some data burned
	   by a known tool */
	if (disc->state_empty == CDW_FALSE) {
		if (disc->cdio->ofs->cdw_tool == CDW_TOOL_XORRISO) {
			if (cdw_ext_tools_is_tool_available(CDW_TOOL_XORRISO)) {

				tool->id = CDW_TOOL_XORRISO;
				cdw_vdm ("INFO: selected xorriso as ISO9660 otf tool, based on existing fs\n\n\n");
				return true;
			}
		} else if (disc->cdio->ofs->cdw_tool == CDW_TOOL_MKISOFS) {
			if (cdw_ext_tools_is_tool_available(CDW_TOOL_MKISOFS)
			    && cdw_ext_tools_is_tool_available(CDW_TOOL_CDRECORD)) {

				tool->id = CDW_TOOL_MKISOFS;
				cdw_vdm ("INFO: selected mkisofs as ISO9660 otf tool, based on existing fs\n\n\n");
				return true;
			}
		} else {
			/* there is an optical file system,
			   but created by unknown tool;
			   fall through to generic algorithm; */
			cdw_vdm ("INFO: disc is non-empty, but original tool is unknown; going to generic ISO9660 ofs tool selection algorithm\n\n\n");
		}
	} else {
		cdw_vdm ("INFO: disc is empty, going to generic ISO9660 ofs tool selection algorithm\n\n\n");
	}


	/* generic algorithm; the code below works in two cases:
	   - when disc is empty;
	   - when disc is not empty, but no tool has been selected; */

	if (cdw_ext_tools_is_tool_available(CDW_TOOL_MKISOFS)
	    && cdw_ext_tools_is_tool_available(CDW_TOOL_CDRECORD)) {

		tool->id = CDW_TOOL_MKISOFS;
		cdw_vdm ("INFO: selected mkisofs as ISO9660 otf tool, based on generic algorithm\n\n\n");
		return true;

	} else if (cdw_ext_tools_is_tool_available(CDW_TOOL_XORRISO)) {

		tool->id = CDW_TOOL_XORRISO;
		cdw_vdm ("INFO: selected xorriso as ISO9660 otf tool, based on generic algorithm\n\n\n");
		return true;

	} else {
		return false;
	}
}





/**
   \brief Select ISO9660 fs creator suitable for given DVD disc

   \date Function's top-level comment reviewed on 2012-02-11
   \date Function's body reviewed on 2012-02-11

   Examine type and state of a DVD \p disc and select appropriate
   tool for creating ISO9660 file system on-the-fly (otf).

   The function should be called only when an automatic selection
   of a tool should be performed, because it uses some logic and
   guessing to get the tool.

   Function sets 'id' field of \p tool.

   \param disc - disc to which to examine
   \param tool - variable, in which to set 'id' field

   \return true if tool id has been set successfully
   \return false if tool id has not been set successfully
*/
bool cdw_ext_tools_set_iso9660_otf_tool_dvd(cdw_disc_t *disc, cdw_id_label_t *tool)
{
	tool->id = CDW_TOOL_NONE;

	/* first check if there is already some data burned
	   by a known tool */
	if (disc->state_empty == CDW_FALSE) {

		if (disc->cdio->ofs->cdw_tool == CDW_TOOL_XORRISO
		    && (cdw_ext_tools_disc_type_supported_by_xorriso(disc->type))) {
			if (cdw_ext_tools_is_tool_available(CDW_TOOL_XORRISO)) {

				tool->id = CDW_TOOL_XORRISO;
				cdw_vdm ("INFO: selected xorriso as ISO9660 otf tool, based on existing fs\n\n\n");
				return true;
			}
		} else if (disc->cdio->ofs->cdw_tool == CDW_TOOL_MKISOFS) {
			if (cdw_ext_tools_is_tool_available(CDW_TOOL_MKISOFS)
			    && (cdw_ext_tools_is_tool_available(CDW_TOOL_CDRECORD)
				|| cdw_ext_tools_is_tool_available(CDW_TOOL_GROWISOFS))) {

				tool->id = CDW_TOOL_MKISOFS;
				cdw_vdm ("INFO: selected mkisofs as ISO9660 otf tool, based on existing fs\n\n\n");
				return true;
			}
		} else {
			/* there is an optical file system,
			   but created by unknown tool;
			   fall through to generic algorithm; */
			cdw_vdm ("INFO: disc is non-empty, but original tool is unknown; going to generic ISO9660 ofs tool selection algorithm\n\n\n");
		}
	} else {
		cdw_vdm ("INFO: disc is empty, going to generic ISO9660 ofs tool selection algorithm\n\n\n");
	}


	/* generic algorithm; the code below works in two cases:
	   - when disc is empty;
	   - when disc is not empty, but no tool has been selected; */
	if (cdw_ext_tools_is_tool_available(CDW_TOOL_MKISOFS)
	    && (cdw_ext_tools_is_tool_available(CDW_TOOL_GROWISOFS)
		||cdw_ext_tools_is_tool_available(CDW_TOOL_CDRECORD))) {

		tool->id = CDW_TOOL_MKISOFS;
		cdw_vdm ("INFO: selected mkisofs as ISO9660 otf tool, based on generic algorithm\n\n\n");
		return true;

	} else if (cdw_ext_tools_is_tool_available(CDW_TOOL_XORRISO)
		   && (cdw_ext_tools_disc_type_supported_by_xorriso(disc->type))) {

		tool->id = CDW_TOOL_XORRISO;
		cdw_vdm ("INFO: selected xorriso as ISO9660 otf tool, based on generic algorithm\n\n\n");
		return true;
	} else {
		return false;
	}
}



/* _sa_ = stand alone */
bool cdw_ext_tools_set_iso9660_sa_tool(cdw_id_label_t *tool)
{
	int impl_ind = 0;
	if (cdw_ext_tools.config.manual_selection) {
		int ind = cdw_ext_tools.config.iso9660_handlers.current_ind;
		tool->id = cdw_ext_tools.config.iso9660_handlers.items[ind].id;
		if (tool->id == CDW_TOOL_NONE) {
			return false;
		}
		impl_ind = cdw_ext_tools_get_manual_selection_ind(tool->id);
		cdw_vdm ("INFO: manual selection, tool id = %lld\n", tool->id);
	} else {
		if (cdw_ext_tools_is_tool_available(CDW_TOOL_MKISOFS)) {
			tool->id = CDW_TOOL_MKISOFS;
		} else if (cdw_ext_tools_is_tool_available(CDW_TOOL_XORRISO)) {
			tool->id = CDW_TOOL_XORRISO;
		} else {
			return false;
		}
		impl_ind = CDW_EXT_TOOLS_SYSTEM_DEFAULT_INSTANCE_IND;
		cdw_vdm ("INFO: auto selection, tool id = %lld, impl ind = %d\n", tool->id, impl_ind);
	}

	cdw_assert (cdw_ext_tools_is_tool_available(tool->id), "ERROR: tool was in dd, but is not available\n");
	tool->label = cdw_ext_tools.tools[tool->id].instances[impl_ind];
	if (tool->label == (char *) NULL) {
		cdw_vdm ("ERROR: fullpath #%d to ISO9660 tool %lld is NULL\n", impl_ind, tool->id);
		return false;
	} else {
		cdw_vdm ("INFO: selected tool %lld / %s\n", tool->id, tool->label);
		return true;
	}
}





bool cdw_ext_tools_set_udf_sa_tool(cdw_id_label_t *tool)
{
	if (cdw_ext_tools_is_tool_available(CDW_TOOL_MKUDFFS)) {
		tool->id = CDW_TOOL_MKUDFFS;
	} else {
		return false;
	}

	int impl_ind = CDW_EXT_TOOLS_SYSTEM_DEFAULT_INSTANCE_IND;
	cdw_assert (cdw_ext_tools_is_tool_available(tool->id), "ERROR: tool was in dd, but is not available\n");
	tool->label = cdw_ext_tools.tools[tool->id].instances[impl_ind];
	if (!tool->label) {
		cdw_vdm ("ERROR: fullpath #%d to UDF tool %lld is NULL\n", impl_ind, tool->id);
		return false;
	} else {
		cdw_vdm ("INFO: selected tool %lld / %s\n", tool->id, tool->label);
		return true;
	}
}







bool cdw_ext_tools_set_erase_tool(cdw_disc_t *disc, cdw_id_label_t *tool)
{
	int impl_ind = 0;
	if (cdw_ext_tools.config.manual_selection) {
		cdw_id_t id = cdw_ext_tools_get_manual_disc_handler_family(disc->cdio->simple_type);
		if (id == CDW_TOOL_CDRECORD) {
			tool->id = CDW_TOOL_CDRECORD;
		} else if (id == CDW_TOOL_GROWISOFS) {
			tool->id = CDW_TOOL_DVD_RW_FORMAT;
		} else if (id == CDW_TOOL_XORRISO) {
			tool->id = CDW_TOOL_XORRISO;
		} else {
			return false;
		}
		cdw_assert (cdw_ext_tools_is_tool_available(tool->id), "ERROR: tool was in dd, but is not available\n");
		impl_ind = cdw_ext_tools_get_manual_selection_ind(tool->id);
	} else {
		if (cdw_ext_tools_disc_type_supported_by_dvd_rw_tools(disc->type)
		    && cdw_ext_tools_dvd_rw_tools_available()) {
			tool->id = CDW_TOOL_DVD_RW_FORMAT;
		} else if (cdw_ext_tools_disc_type_supported_by_cdrecord(disc->type)
			   && cdw_ext_tools_is_tool_available(CDW_TOOL_CDRECORD)) {
			tool->id = CDW_TOOL_CDRECORD;
		} else if (cdw_ext_tools_disc_type_supported_by_xorriso(disc->type)
			   && cdw_ext_tools_is_tool_available(CDW_TOOL_XORRISO)) {
			tool->id = CDW_TOOL_XORRISO;
		} else {
			return false;
		}
		impl_ind = CDW_EXT_TOOLS_SYSTEM_DEFAULT_INSTANCE_IND;
	}

	tool->label = cdw_ext_tools.tools[tool->id].instances[impl_ind];
	if (tool->label == (char *) NULL) {
		cdw_vdm ("ERROR: fullpath #%d to erase tool %lld is NULL\n", impl_ind, tool->id);
		return false;
	} else {
		return true;
	}
}





bool cdw_ext_tools_set_media_info_tool(cdw_disc_t *disc, cdw_id_label_t *tool)
{
	int impl_ind = 0;
	if (cdw_ext_tools.config.manual_selection) {
		cdw_id_t id = cdw_ext_tools_get_manual_disc_handler_family(disc->cdio->simple_type);
		if (id == CDW_TOOL_CDRECORD) {
			tool->id = CDW_TOOL_CDRECORD;
		} else if (id == CDW_TOOL_GROWISOFS) {
			tool->id = CDW_TOOL_DVD_RW_MEDIAINFO;
		} else if (id == CDW_TOOL_XORRISO) {
			tool->id = CDW_TOOL_XORRISO;
		} else {
			return false;
		}
		cdw_assert (cdw_ext_tools_is_tool_available(tool->id), "ERROR: tool was in dd, but is not available\n");
		impl_ind = cdw_ext_tools_get_manual_selection_ind(tool->id);
	} else {
		if (disc->cdio->simple_type == CDW_DISC_SIMPLE_TYPE_DVD
		    && cdw_ext_tools_dvd_rw_tools_available()) {
			tool->id = CDW_TOOL_DVD_RW_MEDIAINFO;
		} else if (cdw_ext_tools_is_tool_available(CDW_TOOL_CDRECORD)) {
			tool->id = CDW_TOOL_CDRECORD;
		} else if (cdw_ext_tools_is_tool_available(CDW_TOOL_XORRISO)) {
			tool->id = CDW_TOOL_XORRISO;
		} else {
			return false;
		}
		impl_ind = CDW_EXT_TOOLS_SYSTEM_DEFAULT_INSTANCE_IND;
	}

	tool->label = cdw_ext_tools.tools[tool->id].instances[impl_ind];
	if (tool->label == (char *) NULL) {
		cdw_vdm ("ERROR: fullpath #%d to media info tool %lld is NULL\n", impl_ind, tool->id);
		return false;
	} else {
		return true;
	}
}





bool cdw_ext_tools_digest_tool_available(void)
{
	return cdw_ext_tools.tools[CDW_TOOL_MD5SUM].available
		|| cdw_ext_tools.tools[CDW_TOOL_SHA1SUM].available
		|| cdw_ext_tools.tools[CDW_TOOL_SHA224SUM].available
		|| cdw_ext_tools.tools[CDW_TOOL_SHA256SUM].available
		|| cdw_ext_tools.tools[CDW_TOOL_SHA384SUM].available
		|| cdw_ext_tools.tools[CDW_TOOL_SHA512SUM].available;
}





bool cdw_ext_tools_is_digest_tool(cdw_id_t tool_id)
{
	if (tool_id == CDW_TOOL_MD5SUM
	    || tool_id == CDW_TOOL_SHA1SUM
	    || tool_id == CDW_TOOL_SHA224SUM
	    || tool_id == CDW_TOOL_SHA256SUM
	    || tool_id == CDW_TOOL_SHA384SUM
	    || tool_id == CDW_TOOL_SHA512SUM) {

		return true;
	} else {
		return false;
	}
}





/**
   Simple getter, since struct cdw_ext_tools is local in this file.
*/
int cdw_ext_tools_get_n_iso9660_handlers(void)
{
	return cdw_ext_tools.config.iso9660_handlers.n_items;
}





/**
   Simple getter, since struct cdw_ext_tools is local in this file.
*/
int cdw_ext_tools_get_n_udf_handlers(void)
{
	return cdw_ext_tools.config.udf_handlers.n_items;
}
