/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2016 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

/**
   \file cdw_file_selector.c

   File implements a component that allows user to select file or
   files from native file system.

   User can use Up/Down arrow keys, Home/End keys, PageUp/PageDown
   keys and Enter key to move in file system listing (just like in any
   file manager, e.g. in Midnight Commander, but with single panel).

   This is just a layer of code on top of cdw_fs_browser_t, with some
   added variables.

   This module isn't the best and cleanest piece of code in this
   project. I have extracted it from cdw_file_manager module. I'm sure
   that this module needs more work. Perhaps some code could be moved
   to cdw_fs_browser, or to client code.

   Perhaps even the name of this module is not the best one.
*/

#include <string.h>
#include <stdlib.h>
#include <ctype.h> /* isprint() */

#include "cdw_debug.h"
#include "cdw_fs.h"
#include "cdw_file_selector.h"
#include "cdw_window.h"
#include "gettext.h"
#include "cdw_string.h"
#include "cdw_widgets.h"





static void cdw_file_selector_driver_print_debug(cdw_file_selector_t *selector, int key);
static void cdw_file_selector_add_return_key(cdw_widget_t *widget, int key);





/**
   \brief Create new file selector widget


   \return pointer to new file selector on success
   \return NULL on failure
*/
cdw_file_selector_t *cdw_file_selector_new(void)
{
	cdw_file_selector_t *selector = malloc(sizeof(cdw_file_selector_t));
	if (!selector) {
		return (cdw_file_selector_t *) NULL;
	}


	/* cdw_fs_get_initial_dirpath() tries to return one of
	   following: cwd, home, /tmp, or /  */
	selector->initial_fullpath = cdw_fs_get_initial_dirpath();
	if (!selector->initial_fullpath) {
		cdw_vdm ("ERROR: failed to create initial fullpath\n");
		cdw_file_selector_delete(&selector);
		return NULL;
	}

	/* Initially, after calling cdw_fs_get_initial_dirpath(),
	   selector->initial_fullpath points to dir, so it is OK to
	   call cdw_fs_check_existing_path() with CDW_FS_DIR. */
	int rv = cdw_fs_check_existing_path(selector->initial_fullpath, R_OK | X_OK, CDW_FS_DIR);
	if (rv != 0) {
		cdw_vdm ("ERROR: failed to validate proposed initial fullpath \"%s\"\n", selector->initial_fullpath);
		cdw_file_selector_delete(&selector);
		return NULL;
	}
	cdw_sdm ("INFO: cdw file manager: initial fullpath initialized as = \"%s\"\n", selector->initial_fullpath);

	cdw_widget_init(&selector->widget);
	selector->widget.self = selector;
	selector->widget.add_return_key = cdw_file_selector_add_return_key; /* Custom function for adding return keys by parent cdw_form. */

	selector->first_selection = true;
	selector->fs_browser = (cdw_fs_browser_t *) NULL;
	selector->window = (WINDOW *) NULL;
	selector->title = (char *) NULL;

	selector->widget.type_id = CDW_WIDGET_ID_FILE_SELECTOR;
	selector->return_on_navigation_keys = false;

	return selector;
}





/**
   \brief Delete file selector widget

   The widget is deallocated and *selector is set to NULL.

   \parent selector - pointer to selector widget
*/
void cdw_file_selector_delete(cdw_file_selector_t **selector)
{
	cdw_assert (selector, "ERROR: passed NULL pointer to the function\n");

	if (!selector || !*selector) {
		return;
	}

	CDW_STRING_DELETE ((*selector)->initial_fullpath);
	cdw_fs_browser_delete(&(*selector)->fs_browser);
	cdw_window_delete(&(*selector)->window);
	CDW_STRING_DELETE ((*selector)->title);

	free(*selector);
	*selector = (cdw_file_selector_t *) NULL;

	return;
}





/**
   \brief Driver function for file selector

   Function interpreting keys pressed in file selector window and driving the file selector accordingly

   Escape/'q'/'Q' keys are interpreted as "close file selector window" keys.

   \param selector - file selector

   \return -1 on errors
   \return last pressed key otherwise; CDW_KEY_ESCAPE is returned for Escape/'q'/'Q'
*/
int cdw_file_selector_driver(cdw_file_selector_t *selector)
{
	cdw_fs_browser_debug_report_current_fullpath(selector->fs_browser, "entering file selector driver");

	int key = 'a';
	while (!cdw_widget_is_return_key(&selector->widget, key)) {

		key = cdw_fs_browser_driver(selector->fs_browser);

		cdw_file_selector_driver_print_debug(selector, key);

		if (key == '?') {
			cdw_fs_browser_help_window(selector->fs_browser);
			cdw_list_display_refresh(selector->fs_browser->display);

		} else if (key == -1) {
			cdw_vdm ("ERROR: file system browser returned -1\n");
			key = -1;
			break;

		} else if (cdw_fs_browser_is_fs_navigation_key(key)) {
			/* This updates the path displayed at the
			   border of file selector window. */
			cdw_file_selector_update_window_decoration(selector);

			if (selector->return_on_navigation_keys) {
				/* For widgets that embed file selector. */
				break;
			} else {
				/* For stand-alone file selector window. */
				key = 'a';
			}
		} else {
			/* Either widget's return key that will be
			   caught by loop condition and handled by
			   widget's parent, or a completely
			   meaningless key that should be ignored. */
			;
		}
	}

	if (key == CDW_KEY_ESCAPE || key == 'q' || key == 'Q') {
		/* 'Q' = quit file selector */
		key = CDW_KEY_ESCAPE;
	}

	cdw_vdm ("INFO: returning key %s\n", cdw_ncurses_key_label(key));
	return key;
}





/**
   \brief Simple wrapper for few lines of code

   Function updates three elements that are displayed on borders of
   file selector's window:
   \li window's title
   \li window's bottom string, which is current dirpath taken from selector->fs_browser
   \li '?' char displayed in upper-right corner of the window

   The most common reason to call this function will be to update
   current dirpath at the bottom of a window after every native file
   system navigation event. But since you update bottom border of the
   window, you have to update other parts of the border as well. This
   function updates every part of the border.

   \param selector - file selector that should have its window border updated
*/
void cdw_file_selector_update_window_decoration(cdw_file_selector_t *selector)
{
	char *dirpath = cdw_fs_browser_get_current_printable_dirpath(selector->fs_browser);
	cdw_assert (dirpath, "ERROR: failed to get current dirpath\n");

	cdw_window_add_strings(selector->window, selector->title, dirpath);
	cdw_window_add_help(selector->window);

	wrefresh(selector->window);
	free(dirpath);
	dirpath = (char *) NULL;

	return;
}





/**
   \brief Create window in which a file selector will be displayed

   Data structure for file selector may exist for long time (and is
   persistent during this time) while it is displayed only
   occasionally.

   You can create a file selector with _new(), then _show() it, then
   _hide() it, then _show() it again, etc., and finally _delete() it.

   Location arguments (begin_y and begin_x) are relative to a window
   in which the selector is shown.

   If \parent is NULL, location is relative to ncurses' screen.
   If \parent is not NULL, location is relative to \parent.

   \param selector
   \param parent
   \param n_lines
   \param n_cols
   \param begin_y
   \param begin_x
   \param colors
   \param title
   \param fullpath

   \return CDW_OK on success
   \return CDW_FAILURE on failure
*/
cdw_rv_t cdw_file_selector_show(cdw_file_selector_t *selector,
				WINDOW *parent,
				int n_lines, int n_cols, int begin_y, int begin_x,
				chtype colors, char const *title,
				char const *fullpath)
{
	cdw_assert (title, "ERROR: \"title\" is NULL\n");

	selector->window = cdw_window_new(parent,
					  n_lines, n_cols, begin_y, begin_x,
					  colors,
					  "", "");
	if (!selector->window) {
		cdw_vdm ("ERROR: failed to create window for file system browser\n");
		return CDW_ERROR;
	}

	if (fullpath) {
		cdw_string_set(&selector->initial_fullpath, fullpath);
	}

	selector->title = strdup(title);

	/* fs browser will occupy whole window, without leaving any margins or borders */
	selector->fs_browser = cdw_fs_browser_new(selector->window, selector->initial_fullpath);
	if (!selector->fs_browser) {
		free(selector->window);
		selector->window = (WINDOW *) NULL;

		cdw_vdm ("ERROR: failed to create file system browser\n");
		return CDW_ERROR;
	}
	/* This will pass the keys to fs_browser that we have just created above. */
	cdw_widget_add_return_keys_pointer(&selector->widget, CDW_KEY_ESCAPE, 0);
	cdw_widget_add_return_keys_pointer(&selector->fs_browser->widget, ' ', KEY_UP, KEY_DOWN, KEY_HOME, KEY_END, KEY_PPAGE, KEY_NPAGE, 0);

	if (selector->first_selection) {
		/* First call to file selector in this session with
		   application, path will be a dirpath returned by
		   cdw_fs_get_initial_dirpath(). */
		cdw_fs_browser_browse_into_dir(selector->fs_browser, selector->initial_fullpath);
	} else {
		/* This is not the first call to file selector, and
		   there was some file selected previously. Try to go
		   to the same file again. */
		cdw_fs_browser_browse_to_file(selector->fs_browser, selector->initial_fullpath);
	}
	cdw_file_selector_update_window_decoration(selector);

	return CDW_OK;
}





void cdw_file_selector_hide(cdw_file_selector_t *selector)
{
	if (!selector) {
		return;
	}

	CDW_STRING_DELETE (selector->title);

	cdw_fs_browser_delete(&selector->fs_browser);
	cdw_window_delete(&selector->window);

	return;
}





void cdw_file_selector_driver_print_debug(cdw_file_selector_t *selector, int key)
{
	char *fp = cdw_fs_browser_get_current_fullpath(selector->fs_browser);

	cdw_vdm ("INFO: fs browser: file_i = %zd, fullpath = \"%s\"\n",
		 selector->fs_browser->display->current_item_ind, fp);

	cdw_vdm("INFO: got %s key in fs browser\n", cdw_ncurses_key_label(key));

	/* I'm using cdw_fs_browser_is_return_key() because file selector doesn't have its own return keys. */
	if (cdw_fs_browser_is_return_key(selector->fs_browser, key)) {
		cdw_vdm ("INFO, returning from browser\n");
	} else if (key == ' ') {
		cdw_vdm ("INFO: selected file = \"%s\"\n", fp);
	} else {
		cdw_sdm ("INFO: looping\n");
	}

	free(fp);
	fp = (char *) NULL;

	return;
}





void cdw_file_selector_refresh(cdw_file_selector_t *selector)
{
	cdw_file_selector_update_window_decoration(selector);
	wrefresh(selector->window);
	cdw_list_display_refresh(selector->fs_browser->display);

	return;
}





/**
	\brief Add return key to self and internal widgets that given widget builds on
*/
void cdw_file_selector_add_return_key(cdw_widget_t *widget, int key)
{
	cdw_assert (widget, "ERROR: \"widget\" argument is NULL\n");
	cdw_assert (widget->n_return_keys < CDW_WIDGET_N_RETURN_KEYS_MAX,
		    "ERROR: there are already %d / %d return keys in the widget, can't add another one\n",
		    widget->n_return_keys, CDW_WIDGET_N_RETURN_KEYS_MAX);
	cdw_assert (key != 0, "ERROR: trying to add key == 0, but 0 is an initializer value\n");

	cdw_widget_add_return_key(widget, key);
	
	if (widget->self) {
		cdw_file_selector_t *file_selector = (cdw_file_selector_t *) widget->self;
		file_selector->fs_browser->widget.add_return_key(&file_selector->fs_browser->widget, key);
	}

	return;
}
