// RUN: mlir-opt %s \
// RUN:  -gpu-lower-to-nvvm-pipeline="cubin-chip=sm_90 cubin-features=+ptx80 opt-level=3" \
// RUN:  | mlir-cpu-runner \
// RUN:   --shared-libs=%mlir_cuda_runtime \
// RUN:   --shared-libs=%mlir_runner_utils \
// RUN:   --entry-point-result=void \
// RUN:  | FileCheck %s

// This does 3 TMA loads with 128B Swizzling : 
//    TMA Load: Matrix-A[0:128][0:64]
//    TMA Load: Matrix-B[0:64][0:64]
//    TMA Load: Matrix-B[64:128][0:64]

// Test swizzling with TMA load
// 128B Swizzle Each numbered cell is 16 byte 
// |-------------------------------|
// | 0 | 1 | 2 | 3 | 4 | 5 | 6 | 7 | 
// | 1 | 0 | 3 | 2 | 5 | 4 | 7 | 6 |
// | 2 | 3 | 0 | 1 | 6 | 7 | 4 | 5 |
// | 3 | 2 | 1 | 0 | 7 | 6 | 5 | 4 | 
// | 4 | 5 | 6 | 7 | 0 | 1 | 2 | 3 |
// | 5 | 4 | 7 | 6 | 1 | 0 | 3 | 2 | 
// | 6 | 7 | 4 | 5 | 2 | 3 | 0 | 1 |
// |-------------------------------| 
// | ... pattern repeats ...       |
// |-------------------------------|


!barrierType = !nvgpu.mbarrier.group<memorySpace = #gpu.address_space<workgroup>>
!tokenType = !nvgpu.mbarrier.token

!lhs = memref<128x64xf16>
!shmemlhs = memref<128x64xf16, 3>
!lhsTensorMap = !nvgpu.tensormap.descriptor<tensor = !shmemlhs, swizzle = swizzle_128b, l2promo=none, oob=zero, interleave=none>

!rhs = memref<64x128xf16>
!shmemrhs = memref<64x128xf16, 3>
!rhsTensorMap = !nvgpu.tensormap.descriptor<tensor = memref<64x64xf16, 3>, swizzle = swizzle_128b, l2promo=none, oob=zero, interleave=none>

module @mymod {
  func.func private @printMemrefF32(memref<*xf32>)
  memref.global "private" @bufferLhsGlobal : !shmemlhs
  memref.global "private" @bufferRhsGlobal : !shmemrhs
  llvm.func @printf(!llvm.ptr, ...) -> i32
  func.func @main() {
    %c32768 = arith.constant 32768 : index
    %c-1_i32 = arith.constant -1 : i32
    %c10000000 = arith.constant 10000000 : index
    %c64 = arith.constant 64 : index
    %c1 = arith.constant 1 : index
    %c32 = arith.constant 32 : index
    %c0 = arith.constant 0 : index
    %c128 = arith.constant 128 : index
    %c8 = arith.constant 8 : index
    
    // Step 1. Allocate host data and initialize it.
    %lhs = memref.alloc() : !lhs
    %rhs = memref.alloc() : !rhs
    %lhs32 = memref.alloc() : memref<128x64xf32>
    %rhs32 = memref.alloc() : memref<64x128xf32>
    scf.for %i = %c0 to %c64 step %c1 {
      scf.for %j = %c0 to %c128 step %c1 {
        %v0 = arith.muli %i, %c128 : index
        %v00 = arith.addi %v0, %j : index     
        %v01 = arith.divui %v00, %c8 : index 
        %v2 = arith.index_cast %v01 : index to i32
        %vR = arith.sitofp %v2 : i32 to f16      
        memref.store %vR, %rhs[%i, %j] : !rhs      
        %vR32 = arith.extf %vR : f16 to f32      
        memref.store %vR32, %rhs32[%i, %j] : memref<64x128xf32>      
        %b0 = arith.muli %j, %c64 : index
        %b00 = arith.addi %b0, %i : index
        %b01 = arith.divui %b00, %c8 : index
        %v1 = arith.index_cast %b01 : index to i32
        %vL = arith.sitofp %v1 : i32 to f16
        memref.store %vL, %lhs[%j, %i] : !lhs
        %vL32 = arith.extf %vL : f16 to f32
        memref.store %vL32, %lhs32[%j, %i] : memref<128x64xf32>
      }
    }
    
    // Step 2. Copy host to device
    %0 = gpu.wait async
    %d_glbmem_lhs, %asyncToken = gpu.alloc async [%0] () : !lhs
    %d_glbmem_rhs, %asyncToken_2 = gpu.alloc async [%0] () : !rhs
    %1 = gpu.memcpy async [%0] %d_glbmem_lhs, %lhs : !lhs, !lhs
    %2 = gpu.memcpy async [%0] %d_glbmem_rhs, %rhs : !rhs, !rhs
    
    // Step 3. Create TMA tensor descriptor
    %d_lhs_unranked = memref.cast %d_glbmem_lhs :!lhs  to memref<*xf16>
    %d_rhs_unranked = memref.cast %d_glbmem_rhs :!rhs  to memref<*xf16>

    %d_lhsTensorMap = nvgpu.tma.create.descriptor %d_lhs_unranked box[%c128, %c64] : memref<*xf16> -> !lhsTensorMap
    %d_rhsTensorMap = nvgpu.tma.create.descriptor %d_rhs_unranked box[%c64, %c64] : memref<*xf16> -> !rhsTensorMap

    // Step 4. Launch a GPU kernel
    gpu.launch blocks(%arg0, %arg1, %arg2) in (%arg6 = %c1, %arg7 = %c1, %arg8 = %c1) threads(%arg3, %arg4, %arg5) in (%arg9 = %c128, %arg10 = %c1, %arg11 = %c1) {
      %5 = gpu.block_dim  x
      %6 = gpu.thread_id  x
      %lhsShmem = memref.get_global @bufferLhsGlobal : !shmemlhs
      %rhsShmem = memref.get_global @bufferRhsGlobal : !shmemrhs
      %rhsShmem1 = memref.subview %rhsShmem[0, 0][64, 64][1, 1] : !shmemrhs to memref<64x64xf16, strided<[128, 1]>, 3>
      %rhsShmem2 = memref.subview %rhsShmem[32, 0][64, 64][1, 1] : !shmemrhs to memref<64x64xf16, strided<[128, 1], offset: 4096>, 3>
    
      // Step 5. Initialize the mbarrier
      %9 = nvgpu.mbarrier.create -> !barrierType
      nvgpu.mbarrier.init %9[%c0], %5 : !barrierType
      %10 = arith.cmpi eq, %6, %c0 : index
      
      // Step 6. First thread does TMA load
      scf.if %10 {
        gpu.printf "[GPU] TMA SIZE %d\0A" %c32768 : index
        nvgpu.tma.async.load %d_lhsTensorMap[%c0, %c0], %9[%c0] to %lhsShmem : !lhsTensorMap, !barrierType -> !shmemlhs
        nvgpu.tma.async.load %d_rhsTensorMap[%c0, %c0], %9[%c0] to %rhsShmem1 : !rhsTensorMap, !barrierType -> memref<64x64xf16, strided<[128, 1]>, 3>
        nvgpu.tma.async.load %d_rhsTensorMap[%c64, %c0], %9[%c0] to %rhsShmem2 : !rhsTensorMap, !barrierType -> memref<64x64xf16, strided<[128, 1], offset: 4096>, 3>
        nvgpu.mbarrier.arrive.expect_tx %9[%c0], %c32768 : !barrierType
      } else {
        nvgpu.mbarrier.arrive.expect_tx %9[%c0], %c0 : !barrierType
      }

      // Step 7. Wait until TMA is done
      %phase_c0 = arith.constant 0 : i1
      nvgpu.mbarrier.try_wait.parity %9[%c0], %phase_c0, %c10000000 : !barrierType

      // Step 8. Print loaded data in 128b swizzled
      scf.if %10 {        
        gpu.printf "===--- Matrix B ---=== %d \n" %c-1_i32 : i32
        scf.for %ii = %c0 to %c64 step %c1 {
          scf.for %j = %c0 to %c128 step %c1 {
            %lhs0 = memref.load %rhsShmem[%ii, %j] : !shmemrhs
            %lhs032 = arith.extf %lhs0: f16 to f32
            gpu.printf "%.0f,   " %lhs032 : f32
          }
          gpu.printf "%d\n" %c-1_i32 : i32
        }
        gpu.printf "===----------------=== %d \n" %c-1_i32 : i32
      }
      gpu.barrier
      gpu.terminator
    }
    return
  }
}


// CHECK: [GPU] TMA SIZE 32768
// CHECK: ===--- Matrix B ---===
// CHECK: 0,   0,   0,   0,   0,   0,   0,   0,   1,   1,   1,   1,   1,   1,   1,   1,   2,   2,   2,   2,   2,   2,   2,   2,   3,   3,   3,   3,   3,   3,   3,   3,   4,   4,   4,   4,   4,   4,   4,   4,   5,   5,   5,   5,   5,   5,   5,   5,   6,   6,   6,   6,   6,   6,   6,   6,   7,   7,   7,   7,   7,   7,   7,   7,   17,   17,   17,   17,   17,   17,   17,   17,   16,   16,   16,   16,   16,   16,   16,   16,   19,   19,   19,   19,   19,   19,   19,   19,   18,   18,   18,   18,   18,   18,   18,   18,   21,   21,   21,   21,   21,   21,   21,   21,   20,   20,   20,   20,   20,   20,   20,   20,   23,   23,   23,   23,   23,   23,   23,   23,   22,   22,   22,   22,   22,   22,   22,   22
// CHECK: 34,   34,   34,   34,   34,   34,   34,   34,   35,   35,   35,   35,   35,   35,   35,   35,   32,   32,   32,   32,   32,   32,   32,   32,   33,   33,   33,   33,   33,   33,   33,   33,   38,   38,   38,   38,   38,   38,   38,   38,   39,   39,   39,   39,   39,   39,   39,   39,   36,   36,   36,   36,   36,   36,   36,   36,   37,   37,   37,   37,   37,   37,   37,   37,   51,   51,   51,   51,   51,   51,   51,   51,   50,   50,   50,   50,   50,   50,   50,   50,   49,   49,   49,   49,   49,   49,   49,   49,   48,   48,   48,   48,   48,   48,   48,   48,   55,   55,   55,   55,   55,   55,   55,   55,   54,   54,   54,   54,   54,   54,   54,   54,   53,   53,   53,   53,   53,   53,   53,   53,   52,   52,   52,   52,   52,   52,   52,   52
// CHECK: 68,   68,   68,   68,   68,   68,   68,   68,   69,   69,   69,   69,   69,   69,   69,   69,   70,   70,   70,   70,   70,   70,   70,   70,   71,   71,   71,   71,   71,   71,   71,   71,   64,   64,   64,   64,   64,   64,   64,   64,   65,   65,   65,   65,   65,   65,   65,   65,   66,   66,   66,   66,   66,   66,   66,   66,   67,   67,   67,   67,   67,   67,   67,   67,   85,   85,   85,   85,   85,   85,   85,   85,   84,   84,   84,   84,   84,   84,   84,   84,   87,   87,   87,   87,   87,   87,   87,   87,   86,   86,   86,   86,   86,   86,   86,   86,   81,   81,   81,   81,   81,   81,   81,   81,   80,   80,   80,   80,   80,   80,   80,   80,   83,   83,   83,   83,   83,   83,   83,   83,   82,   82,   82,   82,   82,   82,   82,   82
// CHECK: 102,   102,   102,   102,   102,   102,   102,   102,   103,   103,   103,   103,   103,   103,   103,   103,   100,   100,   100,   100,   100,   100,   100,   100,   101,   101,   101,   101,   101,   101,   101,   101,   98,   98,   98,   98,   98,   98,   98,   98,   99,   99,   99,   99,   99,   99,   99,   99,   96,   96,   96,   96,   96,   96,   96,   96,   97,   97,   97,   97,   97,   97,   97,   97,   119,   119,   119,   119,   119,   119,   119,   119,   118,   118,   118,   118,   118,   118,   118,   118,   117,   117,   117,   117,   117,   117,   117,   117,   116,   116,   116,   116,   116,   116,   116,   116,   115,   115,   115,   115,   115,   115,   115,   115,   114,   114,   114,   114,   114,   114,   114,   114,   113,   113,   113,   113,   113,   113,   113,   113,   112,   112,   112,   112,   112,   112,   112,   112
// CHECK: 128,   128,   128,   128,   128,   128,   128,   128,   129,   129,   129,   129,   129,   129,   129,   129,   130,   130,   130,   130,   130,   130,   130,   130,   131,   131,   131,   131,   131,   131,   131,   131,   132,   132,   132,   132,   132,   132,   132,   132,   133,   133,   133,   133,   133,   133,   133,   133,   134,   134,   134,   134,   134,   134,   134,   134,   135,   135,   135,   135,   135,   135,   135,   135,   145,   145,   145,   145,   145,   145,   145,   145,   144,   144,   144,   144,   144,   144,   144,   144,   147,   147,   147,   147,   147,   147,   147,   147,   146,   146,   146,   146,   146,   146,   146,   146,   149,   149,   149,   149,   149,   149,   149,   149,   148,   148,   148,   148,   148,   148,   148,   148,   151,   151,   151,   151,   151,   151,   151,   151,   150,   150,   150,   150,   150,   150,   150,   150
// CHECK: 162,   162,   162,   162,   162,   162,   162,   162,   163,   163,   163,   163,   163,   163,   163,   163,   160,   160,   160,   160,   160,   160,   160,   160,   161,   161,   161,   161,   161,   161,   161,   161,   166,   166,   166,   166,   166,   166,   166,   166,   167,   167,   167,   167,   167,   167,   167,   167,   164,   164,   164,   164,   164,   164,   164,   164,   165,   165,   165,   165,   165,   165,   165,   165,   179,   179,   179,   179,   179,   179,   179,   179,   178,   178,   178,   178,   178,   178,   178,   178,   177,   177,   177,   177,   177,   177,   177,   177,   176,   176,   176,   176,   176,   176,   176,   176,   183,   183,   183,   183,   183,   183,   183,   183,   182,   182,   182,   182,   182,   182,   182,   182,   181,   181,   181,   181,   181,   181,   181,   181,   180,   180,   180,   180,   180,   180,   180,   180
// CHECK: 196,   196,   196,   196,   196,   196,   196,   196,   197,   197,   197,   197,   197,   197,   197,   197,   198,   198,   198,   198,   198,   198,   198,   198,   199,   199,   199,   199,   199,   199,   199,   199,   192,   192,   192,   192,   192,   192,   192,   192,   193,   193,   193,   193,   193,   193,   193,   193,   194,   194,   194,   194,   194,   194,   194,   194,   195,   195,   195,   195,   195,   195,   195,   195,   213,   213,   213,   213,   213,   213,   213,   213,   212,   212,   212,   212,   212,   212,   212,   212,   215,   215,   215,   215,   215,   215,   215,   215,   214,   214,   214,   214,   214,   214,   214,   214,   209,   209,   209,   209,   209,   209,   209,   209,   208,   208,   208,   208,   208,   208,   208,   208,   211,   211,   211,   211,   211,   211,   211,   211,   210,   210,   210,   210,   210,   210,   210,   210
// CHECK: 230,   230,   230,   230,   230,   230,   230,   230,   231,   231,   231,   231,   231,   231,   231,   231,   228,   228,   228,   228,   228,   228,   228,   228,   229,   229,   229,   229,   229,   229,   229,   229,   226,   226,   226,   226,   226,   226,   226,   226,   227,   227,   227,   227,   227,   227,   227,   227,   224,   224,   224,   224,   224,   224,   224,   224,   225,   225,   225,   225,   225,   225,   225,   225,   247,   247,   247,   247,   247,   247,   247,   247,   246,   246,   246,   246,   246,   246,   246,   246,   245,   245,   245,   245,   245,   245,   245,   245,   244,   244,   244,   244,   244,   244,   244,   244,   243,   243,   243,   243,   243,   243,   243,   243,   242,   242,   242,   242,   242,   242,   242,   242,   241,   241,   241,   241,   241,   241,   241,   241,   240,   240,   240,   240,   240,   240,   240,   240
// CHECK: 256,   256,   256,   256,   256,   256,   256,   256,   257,   257,   257,   257,   257,   257,   257,   257,   258,   258,   258,   258,   258,   258,   258,   258,   259,   259,   259,   259,   259,   259,   259,   259,   260,   260,   260,   260,   260,   260,   260,   260,   261,   261,   261,   261,   261,   261,   261,   261,   262,   262,   262,   262,   262,   262,   262,   262,   263,   263,   263,   263,   263,   263,   263,   263,   273,   273,   273,   273,   273,   273,   273,   273,   272,   272,   272,   272,   272,   272,   272,   272,   275,   275,   275,   275,   275,   275,   275,   275,   274,   274,   274,   274,   274,   274,   274,   274,   277,   277,   277,   277,   277,   277,   277,   277,   276,   276,   276,   276,   276,   276,   276,   276,   279,   279,   279,   279,   279,   279,   279,   279,   278,   278,   278,   278,   278,   278,   278,   278
// CHECK: 290,   290,   290,   290,   290,   290,   290,   290,   291,   291,   291,   291,   291,   291,   291,   291,   288,   288,   288,   288,   288,   288,   288,   288,   289,   289,   289,   289,   289,   289,   289,   289,   294,   294,   294,   294,   294,   294,   294,   294,   295,   295,   295,   295,   295,   295,   295,   295,   292,   292,   292,   292,   292,   292,   292,   292,   293,   293,   293,   293,   293,   293,   293,   293,   307,   307,   307,   307,   307,   307,   307,   307,   306,   306,   306,   306,   306,   306,   306,   306,   305,   305,   305,   305,   305,   305,   305,   305,   304,   304,   304,   304,   304,   304,   304,   304,   311,   311,   311,   311,   311,   311,   311,   311,   310,   310,   310,   310,   310,   310,   310,   310,   309,   309,   309,   309,   309,   309,   309,   309,   308,   308,   308,   308,   308,   308,   308,   308
// CHECK: 324,   324,   324,   324,   324,   324,   324,   324,   325,   325,   325,   325,   325,   325,   325,   325,   326,   326,   326,   326,   326,   326,   326,   326,   327,   327,   327,   327,   327,   327,   327,   327,   320,   320,   320,   320,   320,   320,   320,   320,   321,   321,   321,   321,   321,   321,   321,   321,   322,   322,   322,   322,   322,   322,   322,   322,   323,   323,   323,   323,   323,   323,   323,   323,   341,   341,   341,   341,   341,   341,   341,   341,   340,   340,   340,   340,   340,   340,   340,   340,   343,   343,   343,   343,   343,   343,   343,   343,   342,   342,   342,   342,   342,   342,   342,   342,   337,   337,   337,   337,   337,   337,   337,   337,   336,   336,   336,   336,   336,   336,   336,   336,   339,   339,   339,   339,   339,   339,   339,   339,   338,   338,   338,   338,   338,   338,   338,   338
// CHECK: 358,   358,   358,   358,   358,   358,   358,   358,   359,   359,   359,   359,   359,   359,   359,   359,   356,   356,   356,   356,   356,   356,   356,   356,   357,   357,   357,   357,   357,   357,   357,   357,   354,   354,   354,   354,   354,   354,   354,   354,   355,   355,   355,   355,   355,   355,   355,   355,   352,   352,   352,   352,   352,   352,   352,   352,   353,   353,   353,   353,   353,   353,   353,   353,   375,   375,   375,   375,   375,   375,   375,   375,   374,   374,   374,   374,   374,   374,   374,   374,   373,   373,   373,   373,   373,   373,   373,   373,   372,   372,   372,   372,   372,   372,   372,   372,   371,   371,   371,   371,   371,   371,   371,   371,   370,   370,   370,   370,   370,   370,   370,   370,   369,   369,   369,   369,   369,   369,   369,   369,   368,   368,   368,   368,   368,   368,   368,   368
// CHECK: 384,   384,   384,   384,   384,   384,   384,   384,   385,   385,   385,   385,   385,   385,   385,   385,   386,   386,   386,   386,   386,   386,   386,   386,   387,   387,   387,   387,   387,   387,   387,   387,   388,   388,   388,   388,   388,   388,   388,   388,   389,   389,   389,   389,   389,   389,   389,   389,   390,   390,   390,   390,   390,   390,   390,   390,   391,   391,   391,   391,   391,   391,   391,   391,   401,   401,   401,   401,   401,   401,   401,   401,   400,   400,   400,   400,   400,   400,   400,   400,   403,   403,   403,   403,   403,   403,   403,   403,   402,   402,   402,   402,   402,   402,   402,   402,   405,   405,   405,   405,   405,   405,   405,   405,   404,   404,   404,   404,   404,   404,   404,   404,   407,   407,   407,   407,   407,   407,   407,   407,   406,   406,   406,   406,   406,   406,   406,   406
// CHECK: 418,   418,   418,   418,   418,   418,   418,   418,   419,   419,   419,   419,   419,   419,   419,   419,   416,   416,   416,   416,   416,   416,   416,   416,   417,   417,   417,   417,   417,   417,   417,   417,   422,   422,   422,   422,   422,   422,   422,   422,   423,   423,   423,   423,   423,   423,   423,   423,   420,   420,   420,   420,   420,   420,   420,   420,   421,   421,   421,   421,   421,   421,   421,   421,   435,   435,   435,   435,   435,   435,   435,   435,   434,   434,   434,   434,   434,   434,   434,   434,   433,   433,   433,   433,   433,   433,   433,   433,   432,   432,   432,   432,   432,   432,   432,   432,   439,   439,   439,   439,   439,   439,   439,   439,   438,   438,   438,   438,   438,   438,   438,   438,   437,   437,   437,   437,   437,   437,   437,   437,   436,   436,   436,   436,   436,   436,   436,   436
// CHECK: 452,   452,   452,   452,   452,   452,   452,   452,   453,   453,   453,   453,   453,   453,   453,   453,   454,   454,   454,   454,   454,   454,   454,   454,   455,   455,   455,   455,   455,   455,   455,   455,   448,   448,   448,   448,   448,   448,   448,   448,   449,   449,   449,   449,   449,   449,   449,   449,   450,   450,   450,   450,   450,   450,   450,   450,   451,   451,   451,   451,   451,   451,   451,   451,   469,   469,   469,   469,   469,   469,   469,   469,   468,   468,   468,   468,   468,   468,   468,   468,   471,   471,   471,   471,   471,   471,   471,   471,   470,   470,   470,   470,   470,   470,   470,   470,   465,   465,   465,   465,   465,   465,   465,   465,   464,   464,   464,   464,   464,   464,   464,   464,   467,   467,   467,   467,   467,   467,   467,   467,   466,   466,   466,   466,   466,   466,   466,   466
// CHECK: 486,   486,   486,   486,   486,   486,   486,   486,   487,   487,   487,   487,   487,   487,   487,   487,   484,   484,   484,   484,   484,   484,   484,   484,   485,   485,   485,   485,   485,   485,   485,   485,   482,   482,   482,   482,   482,   482,   482,   482,   483,   483,   483,   483,   483,   483,   483,   483,   480,   480,   480,   480,   480,   480,   480,   480,   481,   481,   481,   481,   481,   481,   481,   481,   503,   503,   503,   503,   503,   503,   503,   503,   502,   502,   502,   502,   502,   502,   502,   502,   501,   501,   501,   501,   501,   501,   501,   501,   500,   500,   500,   500,   500,   500,   500,   500,   499,   499,   499,   499,   499,   499,   499,   499,   498,   498,   498,   498,   498,   498,   498,   498,   497,   497,   497,   497,   497,   497,   497,   497,   496,   496,   496,   496,   496,   496,   496,   496
// CHECK: 512,   512,   512,   512,   512,   512,   512,   512,   513,   513,   513,   513,   513,   513,   513,   513,   514,   514,   514,   514,   514,   514,   514,   514,   515,   515,   515,   515,   515,   515,   515,   515,   516,   516,   516,   516,   516,   516,   516,   516,   517,   517,   517,   517,   517,   517,   517,   517,   518,   518,   518,   518,   518,   518,   518,   518,   519,   519,   519,   519,   519,   519,   519,   519,   529,   529,   529,   529,   529,   529,   529,   529,   528,   528,   528,   528,   528,   528,   528,   528,   531,   531,   531,   531,   531,   531,   531,   531,   530,   530,   530,   530,   530,   530,   530,   530,   533,   533,   533,   533,   533,   533,   533,   533,   532,   532,   532,   532,   532,   532,   532,   532,   535,   535,   535,   535,   535,   535,   535,   535,   534,   534,   534,   534,   534,   534,   534,   534
// CHECK: 546,   546,   546,   546,   546,   546,   546,   546,   547,   547,   547,   547,   547,   547,   547,   547,   544,   544,   544,   544,   544,   544,   544,   544,   545,   545,   545,   545,   545,   545,   545,   545,   550,   550,   550,   550,   550,   550,   550,   550,   551,   551,   551,   551,   551,   551,   551,   551,   548,   548,   548,   548,   548,   548,   548,   548,   549,   549,   549,   549,   549,   549,   549,   549,   563,   563,   563,   563,   563,   563,   563,   563,   562,   562,   562,   562,   562,   562,   562,   562,   561,   561,   561,   561,   561,   561,   561,   561,   560,   560,   560,   560,   560,   560,   560,   560,   567,   567,   567,   567,   567,   567,   567,   567,   566,   566,   566,   566,   566,   566,   566,   566,   565,   565,   565,   565,   565,   565,   565,   565,   564,   564,   564,   564,   564,   564,   564,   564
// CHECK: 580,   580,   580,   580,   580,   580,   580,   580,   581,   581,   581,   581,   581,   581,   581,   581,   582,   582,   582,   582,   582,   582,   582,   582,   583,   583,   583,   583,   583,   583,   583,   583,   576,   576,   576,   576,   576,   576,   576,   576,   577,   577,   577,   577,   577,   577,   577,   577,   578,   578,   578,   578,   578,   578,   578,   578,   579,   579,   579,   579,   579,   579,   579,   579,   597,   597,   597,   597,   597,   597,   597,   597,   596,   596,   596,   596,   596,   596,   596,   596,   599,   599,   599,   599,   599,   599,   599,   599,   598,   598,   598,   598,   598,   598,   598,   598,   593,   593,   593,   593,   593,   593,   593,   593,   592,   592,   592,   592,   592,   592,   592,   592,   595,   595,   595,   595,   595,   595,   595,   595,   594,   594,   594,   594,   594,   594,   594,   594
// CHECK: 614,   614,   614,   614,   614,   614,   614,   614,   615,   615,   615,   615,   615,   615,   615,   615,   612,   612,   612,   612,   612,   612,   612,   612,   613,   613,   613,   613,   613,   613,   613,   613,   610,   610,   610,   610,   610,   610,   610,   610,   611,   611,   611,   611,   611,   611,   611,   611,   608,   608,   608,   608,   608,   608,   608,   608,   609,   609,   609,   609,   609,   609,   609,   609,   631,   631,   631,   631,   631,   631,   631,   631,   630,   630,   630,   630,   630,   630,   630,   630,   629,   629,   629,   629,   629,   629,   629,   629,   628,   628,   628,   628,   628,   628,   628,   628,   627,   627,   627,   627,   627,   627,   627,   627,   626,   626,   626,   626,   626,   626,   626,   626,   625,   625,   625,   625,   625,   625,   625,   625,   624,   624,   624,   624,   624,   624,   624,   624
// CHECK: 640,   640,   640,   640,   640,   640,   640,   640,   641,   641,   641,   641,   641,   641,   641,   641,   642,   642,   642,   642,   642,   642,   642,   642,   643,   643,   643,   643,   643,   643,   643,   643,   644,   644,   644,   644,   644,   644,   644,   644,   645,   645,   645,   645,   645,   645,   645,   645,   646,   646,   646,   646,   646,   646,   646,   646,   647,   647,   647,   647,   647,   647,   647,   647,   657,   657,   657,   657,   657,   657,   657,   657,   656,   656,   656,   656,   656,   656,   656,   656,   659,   659,   659,   659,   659,   659,   659,   659,   658,   658,   658,   658,   658,   658,   658,   658,   661,   661,   661,   661,   661,   661,   661,   661,   660,   660,   660,   660,   660,   660,   660,   660,   663,   663,   663,   663,   663,   663,   663,   663,   662,   662,   662,   662,   662,   662,   662,   662
// CHECK: 674,   674,   674,   674,   674,   674,   674,   674,   675,   675,   675,   675,   675,   675,   675,   675,   672,   672,   672,   672,   672,   672,   672,   672,   673,   673,   673,   673,   673,   673,   673,   673,   678,   678,   678,   678,   678,   678,   678,   678,   679,   679,   679,   679,   679,   679,   679,   679,   676,   676,   676,   676,   676,   676,   676,   676,   677,   677,   677,   677,   677,   677,   677,   677,   691,   691,   691,   691,   691,   691,   691,   691,   690,   690,   690,   690,   690,   690,   690,   690,   689,   689,   689,   689,   689,   689,   689,   689,   688,   688,   688,   688,   688,   688,   688,   688,   695,   695,   695,   695,   695,   695,   695,   695,   694,   694,   694,   694,   694,   694,   694,   694,   693,   693,   693,   693,   693,   693,   693,   693,   692,   692,   692,   692,   692,   692,   692,   692
// CHECK: 708,   708,   708,   708,   708,   708,   708,   708,   709,   709,   709,   709,   709,   709,   709,   709,   710,   710,   710,   710,   710,   710,   710,   710,   711,   711,   711,   711,   711,   711,   711,   711,   704,   704,   704,   704,   704,   704,   704,   704,   705,   705,   705,   705,   705,   705,   705,   705,   706,   706,   706,   706,   706,   706,   706,   706,   707,   707,   707,   707,   707,   707,   707,   707,   725,   725,   725,   725,   725,   725,   725,   725,   724,   724,   724,   724,   724,   724,   724,   724,   727,   727,   727,   727,   727,   727,   727,   727,   726,   726,   726,   726,   726,   726,   726,   726,   721,   721,   721,   721,   721,   721,   721,   721,   720,   720,   720,   720,   720,   720,   720,   720,   723,   723,   723,   723,   723,   723,   723,   723,   722,   722,   722,   722,   722,   722,   722,   722
// CHECK: 742,   742,   742,   742,   742,   742,   742,   742,   743,   743,   743,   743,   743,   743,   743,   743,   740,   740,   740,   740,   740,   740,   740,   740,   741,   741,   741,   741,   741,   741,   741,   741,   738,   738,   738,   738,   738,   738,   738,   738,   739,   739,   739,   739,   739,   739,   739,   739,   736,   736,   736,   736,   736,   736,   736,   736,   737,   737,   737,   737,   737,   737,   737,   737,   759,   759,   759,   759,   759,   759,   759,   759,   758,   758,   758,   758,   758,   758,   758,   758,   757,   757,   757,   757,   757,   757,   757,   757,   756,   756,   756,   756,   756,   756,   756,   756,   755,   755,   755,   755,   755,   755,   755,   755,   754,   754,   754,   754,   754,   754,   754,   754,   753,   753,   753,   753,   753,   753,   753,   753,   752,   752,   752,   752,   752,   752,   752,   752
// CHECK: 768,   768,   768,   768,   768,   768,   768,   768,   769,   769,   769,   769,   769,   769,   769,   769,   770,   770,   770,   770,   770,   770,   770,   770,   771,   771,   771,   771,   771,   771,   771,   771,   772,   772,   772,   772,   772,   772,   772,   772,   773,   773,   773,   773,   773,   773,   773,   773,   774,   774,   774,   774,   774,   774,   774,   774,   775,   775,   775,   775,   775,   775,   775,   775,   785,   785,   785,   785,   785,   785,   785,   785,   784,   784,   784,   784,   784,   784,   784,   784,   787,   787,   787,   787,   787,   787,   787,   787,   786,   786,   786,   786,   786,   786,   786,   786,   789,   789,   789,   789,   789,   789,   789,   789,   788,   788,   788,   788,   788,   788,   788,   788,   791,   791,   791,   791,   791,   791,   791,   791,   790,   790,   790,   790,   790,   790,   790,   790
// CHECK: 802,   802,   802,   802,   802,   802,   802,   802,   803,   803,   803,   803,   803,   803,   803,   803,   800,   800,   800,   800,   800,   800,   800,   800,   801,   801,   801,   801,   801,   801,   801,   801,   806,   806,   806,   806,   806,   806,   806,   806,   807,   807,   807,   807,   807,   807,   807,   807,   804,   804,   804,   804,   804,   804,   804,   804,   805,   805,   805,   805,   805,   805,   805,   805,   819,   819,   819,   819,   819,   819,   819,   819,   818,   818,   818,   818,   818,   818,   818,   818,   817,   817,   817,   817,   817,   817,   817,   817,   816,   816,   816,   816,   816,   816,   816,   816,   823,   823,   823,   823,   823,   823,   823,   823,   822,   822,   822,   822,   822,   822,   822,   822,   821,   821,   821,   821,   821,   821,   821,   821,   820,   820,   820,   820,   820,   820,   820,   820
// CHECK: 836,   836,   836,   836,   836,   836,   836,   836,   837,   837,   837,   837,   837,   837,   837,   837,   838,   838,   838,   838,   838,   838,   838,   838,   839,   839,   839,   839,   839,   839,   839,   839,   832,   832,   832,   832,   832,   832,   832,   832,   833,   833,   833,   833,   833,   833,   833,   833,   834,   834,   834,   834,   834,   834,   834,   834,   835,   835,   835,   835,   835,   835,   835,   835,   853,   853,   853,   853,   853,   853,   853,   853,   852,   852,   852,   852,   852,   852,   852,   852,   855,   855,   855,   855,   855,   855,   855,   855,   854,   854,   854,   854,   854,   854,   854,   854,   849,   849,   849,   849,   849,   849,   849,   849,   848,   848,   848,   848,   848,   848,   848,   848,   851,   851,   851,   851,   851,   851,   851,   851,   850,   850,   850,   850,   850,   850,   850,   850
// CHECK: 870,   870,   870,   870,   870,   870,   870,   870,   871,   871,   871,   871,   871,   871,   871,   871,   868,   868,   868,   868,   868,   868,   868,   868,   869,   869,   869,   869,   869,   869,   869,   869,   866,   866,   866,   866,   866,   866,   866,   866,   867,   867,   867,   867,   867,   867,   867,   867,   864,   864,   864,   864,   864,   864,   864,   864,   865,   865,   865,   865,   865,   865,   865,   865,   887,   887,   887,   887,   887,   887,   887,   887,   886,   886,   886,   886,   886,   886,   886,   886,   885,   885,   885,   885,   885,   885,   885,   885,   884,   884,   884,   884,   884,   884,   884,   884,   883,   883,   883,   883,   883,   883,   883,   883,   882,   882,   882,   882,   882,   882,   882,   882,   881,   881,   881,   881,   881,   881,   881,   881,   880,   880,   880,   880,   880,   880,   880,   880
// CHECK: 896,   896,   896,   896,   896,   896,   896,   896,   897,   897,   897,   897,   897,   897,   897,   897,   898,   898,   898,   898,   898,   898,   898,   898,   899,   899,   899,   899,   899,   899,   899,   899,   900,   900,   900,   900,   900,   900,   900,   900,   901,   901,   901,   901,   901,   901,   901,   901,   902,   902,   902,   902,   902,   902,   902,   902,   903,   903,   903,   903,   903,   903,   903,   903,   913,   913,   913,   913,   913,   913,   913,   913,   912,   912,   912,   912,   912,   912,   912,   912,   915,   915,   915,   915,   915,   915,   915,   915,   914,   914,   914,   914,   914,   914,   914,   914,   917,   917,   917,   917,   917,   917,   917,   917,   916,   916,   916,   916,   916,   916,   916,   916,   919,   919,   919,   919,   919,   919,   919,   919,   918,   918,   918,   918,   918,   918,   918,   918
// CHECK: 930,   930,   930,   930,   930,   930,   930,   930,   931,   931,   931,   931,   931,   931,   931,   931,   928,   928,   928,   928,   928,   928,   928,   928,   929,   929,   929,   929,   929,   929,   929,   929,   934,   934,   934,   934,   934,   934,   934,   934,   935,   935,   935,   935,   935,   935,   935,   935,   932,   932,   932,   932,   932,   932,   932,   932,   933,   933,   933,   933,   933,   933,   933,   933,   947,   947,   947,   947,   947,   947,   947,   947,   946,   946,   946,   946,   946,   946,   946,   946,   945,   945,   945,   945,   945,   945,   945,   945,   944,   944,   944,   944,   944,   944,   944,   944,   951,   951,   951,   951,   951,   951,   951,   951,   950,   950,   950,   950,   950,   950,   950,   950,   949,   949,   949,   949,   949,   949,   949,   949,   948,   948,   948,   948,   948,   948,   948,   948
// CHECK: 964,   964,   964,   964,   964,   964,   964,   964,   965,   965,   965,   965,   965,   965,   965,   965,   966,   966,   966,   966,   966,   966,   966,   966,   967,   967,   967,   967,   967,   967,   967,   967,   960,   960,   960,   960,   960,   960,   960,   960,   961,   961,   961,   961,   961,   961,   961,   961,   962,   962,   962,   962,   962,   962,   962,   962,   963,   963,   963,   963,   963,   963,   963,   963,   981,   981,   981,   981,   981,   981,   981,   981,   980,   980,   980,   980,   980,   980,   980,   980,   983,   983,   983,   983,   983,   983,   983,   983,   982,   982,   982,   982,   982,   982,   982,   982,   977,   977,   977,   977,   977,   977,   977,   977,   976,   976,   976,   976,   976,   976,   976,   976,   979,   979,   979,   979,   979,   979,   979,   979,   978,   978,   978,   978,   978,   978,   978,   978
// CHECK: 998,   998,   998,   998,   998,   998,   998,   998,   999,   999,   999,   999,   999,   999,   999,   999,   996,   996,   996,   996,   996,   996,   996,   996,   997,   997,   997,   997,   997,   997,   997,   997,   994,   994,   994,   994,   994,   994,   994,   994,   995,   995,   995,   995,   995,   995,   995,   995,   992,   992,   992,   992,   992,   992,   992,   992,   993,   993,   993,   993,   993,   993,   993,   993,   1015,   1015,   1015,   1015,   1015,   1015,   1015,   1015,   1014,   1014,   1014,   1014,   1014,   1014,   1014,   1014,   1013,   1013,   1013,   1013,   1013,   1013,   1013,   1013,   1012,   1012,   1012,   1012,   1012,   1012,   1012,   1012,   1011,   1011,   1011,   1011,   1011,   1011,   1011,   1011,   1010,   1010,   1010,   1010,   1010,   1010,   1010,   1010,   1009,   1009,   1009,   1009,   1009,   1009,   1009,   1009,   1008,   1008,   1008,   1008,   1008,   1008,   1008,   1008
// CHECK: 8,   8,   8,   8,   8,   8,   8,   8,   9,   9,   9,   9,   9,   9,   9,   9,   10,   10,   10,   10,   10,   10,   10,   10,   11,   11,   11,   11,   11,   11,   11,   11,   12,   12,   12,   12,   12,   12,   12,   12,   13,   13,   13,   13,   13,   13,   13,   13,   14,   14,   14,   14,   14,   14,   14,   14,   15,   15,   15,   15,   15,   15,   15,   15,   25,   25,   25,   25,   25,   25,   25,   25,   24,   24,   24,   24,   24,   24,   24,   24,   27,   27,   27,   27,   27,   27,   27,   27,   26,   26,   26,   26,   26,   26,   26,   26,   29,   29,   29,   29,   29,   29,   29,   29,   28,   28,   28,   28,   28,   28,   28,   28,   31,   31,   31,   31,   31,   31,   31,   31,   30,   30,   30,   30,   30,   30,   30,   30
// CHECK: 42,   42,   42,   42,   42,   42,   42,   42,   43,   43,   43,   43,   43,   43,   43,   43,   40,   40,   40,   40,   40,   40,   40,   40,   41,   41,   41,   41,   41,   41,   41,   41,   46,   46,   46,   46,   46,   46,   46,   46,   47,   47,   47,   47,   47,   47,   47,   47,   44,   44,   44,   44,   44,   44,   44,   44,   45,   45,   45,   45,   45,   45,   45,   45,   59,   59,   59,   59,   59,   59,   59,   59,   58,   58,   58,   58,   58,   58,   58,   58,   57,   57,   57,   57,   57,   57,   57,   57,   56,   56,   56,   56,   56,   56,   56,   56,   63,   63,   63,   63,   63,   63,   63,   63,   62,   62,   62,   62,   62,   62,   62,   62,   61,   61,   61,   61,   61,   61,   61,   61,   60,   60,   60,   60,   60,   60,   60,   60
// CHECK: 76,   76,   76,   76,   76,   76,   76,   76,   77,   77,   77,   77,   77,   77,   77,   77,   78,   78,   78,   78,   78,   78,   78,   78,   79,   79,   79,   79,   79,   79,   79,   79,   72,   72,   72,   72,   72,   72,   72,   72,   73,   73,   73,   73,   73,   73,   73,   73,   74,   74,   74,   74,   74,   74,   74,   74,   75,   75,   75,   75,   75,   75,   75,   75,   93,   93,   93,   93,   93,   93,   93,   93,   92,   92,   92,   92,   92,   92,   92,   92,   95,   95,   95,   95,   95,   95,   95,   95,   94,   94,   94,   94,   94,   94,   94,   94,   89,   89,   89,   89,   89,   89,   89,   89,   88,   88,   88,   88,   88,   88,   88,   88,   91,   91,   91,   91,   91,   91,   91,   91,   90,   90,   90,   90,   90,   90,   90,   90
// CHECK: 110,   110,   110,   110,   110,   110,   110,   110,   111,   111,   111,   111,   111,   111,   111,   111,   108,   108,   108,   108,   108,   108,   108,   108,   109,   109,   109,   109,   109,   109,   109,   109,   106,   106,   106,   106,   106,   106,   106,   106,   107,   107,   107,   107,   107,   107,   107,   107,   104,   104,   104,   104,   104,   104,   104,   104,   105,   105,   105,   105,   105,   105,   105,   105,   127,   127,   127,   127,   127,   127,   127,   127,   126,   126,   126,   126,   126,   126,   126,   126,   125,   125,   125,   125,   125,   125,   125,   125,   124,   124,   124,   124,   124,   124,   124,   124,   123,   123,   123,   123,   123,   123,   123,   123,   122,   122,   122,   122,   122,   122,   122,   122,   121,   121,   121,   121,   121,   121,   121,   121,   120,   120,   120,   120,   120,   120,   120,   120
// CHECK: 136,   136,   136,   136,   136,   136,   136,   136,   137,   137,   137,   137,   137,   137,   137,   137,   138,   138,   138,   138,   138,   138,   138,   138,   139,   139,   139,   139,   139,   139,   139,   139,   140,   140,   140,   140,   140,   140,   140,   140,   141,   141,   141,   141,   141,   141,   141,   141,   142,   142,   142,   142,   142,   142,   142,   142,   143,   143,   143,   143,   143,   143,   143,   143,   153,   153,   153,   153,   153,   153,   153,   153,   152,   152,   152,   152,   152,   152,   152,   152,   155,   155,   155,   155,   155,   155,   155,   155,   154,   154,   154,   154,   154,   154,   154,   154,   157,   157,   157,   157,   157,   157,   157,   157,   156,   156,   156,   156,   156,   156,   156,   156,   159,   159,   159,   159,   159,   159,   159,   159,   158,   158,   158,   158,   158,   158,   158,   158
// CHECK: 170,   170,   170,   170,   170,   170,   170,   170,   171,   171,   171,   171,   171,   171,   171,   171,   168,   168,   168,   168,   168,   168,   168,   168,   169,   169,   169,   169,   169,   169,   169,   169,   174,   174,   174,   174,   174,   174,   174,   174,   175,   175,   175,   175,   175,   175,   175,   175,   172,   172,   172,   172,   172,   172,   172,   172,   173,   173,   173,   173,   173,   173,   173,   173,   187,   187,   187,   187,   187,   187,   187,   187,   186,   186,   186,   186,   186,   186,   186,   186,   185,   185,   185,   185,   185,   185,   185,   185,   184,   184,   184,   184,   184,   184,   184,   184,   191,   191,   191,   191,   191,   191,   191,   191,   190,   190,   190,   190,   190,   190,   190,   190,   189,   189,   189,   189,   189,   189,   189,   189,   188,   188,   188,   188,   188,   188,   188,   188
// CHECK: 204,   204,   204,   204,   204,   204,   204,   204,   205,   205,   205,   205,   205,   205,   205,   205,   206,   206,   206,   206,   206,   206,   206,   206,   207,   207,   207,   207,   207,   207,   207,   207,   200,   200,   200,   200,   200,   200,   200,   200,   201,   201,   201,   201,   201,   201,   201,   201,   202,   202,   202,   202,   202,   202,   202,   202,   203,   203,   203,   203,   203,   203,   203,   203,   221,   221,   221,   221,   221,   221,   221,   221,   220,   220,   220,   220,   220,   220,   220,   220,   223,   223,   223,   223,   223,   223,   223,   223,   222,   222,   222,   222,   222,   222,   222,   222,   217,   217,   217,   217,   217,   217,   217,   217,   216,   216,   216,   216,   216,   216,   216,   216,   219,   219,   219,   219,   219,   219,   219,   219,   218,   218,   218,   218,   218,   218,   218,   218
// CHECK: 238,   238,   238,   238,   238,   238,   238,   238,   239,   239,   239,   239,   239,   239,   239,   239,   236,   236,   236,   236,   236,   236,   236,   236,   237,   237,   237,   237,   237,   237,   237,   237,   234,   234,   234,   234,   234,   234,   234,   234,   235,   235,   235,   235,   235,   235,   235,   235,   232,   232,   232,   232,   232,   232,   232,   232,   233,   233,   233,   233,   233,   233,   233,   233,   255,   255,   255,   255,   255,   255,   255,   255,   254,   254,   254,   254,   254,   254,   254,   254,   253,   253,   253,   253,   253,   253,   253,   253,   252,   252,   252,   252,   252,   252,   252,   252,   251,   251,   251,   251,   251,   251,   251,   251,   250,   250,   250,   250,   250,   250,   250,   250,   249,   249,   249,   249,   249,   249,   249,   249,   248,   248,   248,   248,   248,   248,   248,   248
// CHECK: 264,   264,   264,   264,   264,   264,   264,   264,   265,   265,   265,   265,   265,   265,   265,   265,   266,   266,   266,   266,   266,   266,   266,   266,   267,   267,   267,   267,   267,   267,   267,   267,   268,   268,   268,   268,   268,   268,   268,   268,   269,   269,   269,   269,   269,   269,   269,   269,   270,   270,   270,   270,   270,   270,   270,   270,   271,   271,   271,   271,   271,   271,   271,   271,   281,   281,   281,   281,   281,   281,   281,   281,   280,   280,   280,   280,   280,   280,   280,   280,   283,   283,   283,   283,   283,   283,   283,   283,   282,   282,   282,   282,   282,   282,   282,   282,   285,   285,   285,   285,   285,   285,   285,   285,   284,   284,   284,   284,   284,   284,   284,   284,   287,   287,   287,   287,   287,   287,   287,   287,   286,   286,   286,   286,   286,   286,   286,   286
// CHECK: 298,   298,   298,   298,   298,   298,   298,   298,   299,   299,   299,   299,   299,   299,   299,   299,   296,   296,   296,   296,   296,   296,   296,   296,   297,   297,   297,   297,   297,   297,   297,   297,   302,   302,   302,   302,   302,   302,   302,   302,   303,   303,   303,   303,   303,   303,   303,   303,   300,   300,   300,   300,   300,   300,   300,   300,   301,   301,   301,   301,   301,   301,   301,   301,   315,   315,   315,   315,   315,   315,   315,   315,   314,   314,   314,   314,   314,   314,   314,   314,   313,   313,   313,   313,   313,   313,   313,   313,   312,   312,   312,   312,   312,   312,   312,   312,   319,   319,   319,   319,   319,   319,   319,   319,   318,   318,   318,   318,   318,   318,   318,   318,   317,   317,   317,   317,   317,   317,   317,   317,   316,   316,   316,   316,   316,   316,   316,   316
// CHECK: 332,   332,   332,   332,   332,   332,   332,   332,   333,   333,   333,   333,   333,   333,   333,   333,   334,   334,   334,   334,   334,   334,   334,   334,   335,   335,   335,   335,   335,   335,   335,   335,   328,   328,   328,   328,   328,   328,   328,   328,   329,   329,   329,   329,   329,   329,   329,   329,   330,   330,   330,   330,   330,   330,   330,   330,   331,   331,   331,   331,   331,   331,   331,   331,   349,   349,   349,   349,   349,   349,   349,   349,   348,   348,   348,   348,   348,   348,   348,   348,   351,   351,   351,   351,   351,   351,   351,   351,   350,   350,   350,   350,   350,   350,   350,   350,   345,   345,   345,   345,   345,   345,   345,   345,   344,   344,   344,   344,   344,   344,   344,   344,   347,   347,   347,   347,   347,   347,   347,   347,   346,   346,   346,   346,   346,   346,   346,   346
// CHECK: 366,   366,   366,   366,   366,   366,   366,   366,   367,   367,   367,   367,   367,   367,   367,   367,   364,   364,   364,   364,   364,   364,   364,   364,   365,   365,   365,   365,   365,   365,   365,   365,   362,   362,   362,   362,   362,   362,   362,   362,   363,   363,   363,   363,   363,   363,   363,   363,   360,   360,   360,   360,   360,   360,   360,   360,   361,   361,   361,   361,   361,   361,   361,   361,   383,   383,   383,   383,   383,   383,   383,   383,   382,   382,   382,   382,   382,   382,   382,   382,   381,   381,   381,   381,   381,   381,   381,   381,   380,   380,   380,   380,   380,   380,   380,   380,   379,   379,   379,   379,   379,   379,   379,   379,   378,   378,   378,   378,   378,   378,   378,   378,   377,   377,   377,   377,   377,   377,   377,   377,   376,   376,   376,   376,   376,   376,   376,   376
// CHECK: 392,   392,   392,   392,   392,   392,   392,   392,   393,   393,   393,   393,   393,   393,   393,   393,   394,   394,   394,   394,   394,   394,   394,   394,   395,   395,   395,   395,   395,   395,   395,   395,   396,   396,   396,   396,   396,   396,   396,   396,   397,   397,   397,   397,   397,   397,   397,   397,   398,   398,   398,   398,   398,   398,   398,   398,   399,   399,   399,   399,   399,   399,   399,   399,   409,   409,   409,   409,   409,   409,   409,   409,   408,   408,   408,   408,   408,   408,   408,   408,   411,   411,   411,   411,   411,   411,   411,   411,   410,   410,   410,   410,   410,   410,   410,   410,   413,   413,   413,   413,   413,   413,   413,   413,   412,   412,   412,   412,   412,   412,   412,   412,   415,   415,   415,   415,   415,   415,   415,   415,   414,   414,   414,   414,   414,   414,   414,   414
// CHECK: 426,   426,   426,   426,   426,   426,   426,   426,   427,   427,   427,   427,   427,   427,   427,   427,   424,   424,   424,   424,   424,   424,   424,   424,   425,   425,   425,   425,   425,   425,   425,   425,   430,   430,   430,   430,   430,   430,   430,   430,   431,   431,   431,   431,   431,   431,   431,   431,   428,   428,   428,   428,   428,   428,   428,   428,   429,   429,   429,   429,   429,   429,   429,   429,   443,   443,   443,   443,   443,   443,   443,   443,   442,   442,   442,   442,   442,   442,   442,   442,   441,   441,   441,   441,   441,   441,   441,   441,   440,   440,   440,   440,   440,   440,   440,   440,   447,   447,   447,   447,   447,   447,   447,   447,   446,   446,   446,   446,   446,   446,   446,   446,   445,   445,   445,   445,   445,   445,   445,   445,   444,   444,   444,   444,   444,   444,   444,   444
// CHECK: 460,   460,   460,   460,   460,   460,   460,   460,   461,   461,   461,   461,   461,   461,   461,   461,   462,   462,   462,   462,   462,   462,   462,   462,   463,   463,   463,   463,   463,   463,   463,   463,   456,   456,   456,   456,   456,   456,   456,   456,   457,   457,   457,   457,   457,   457,   457,   457,   458,   458,   458,   458,   458,   458,   458,   458,   459,   459,   459,   459,   459,   459,   459,   459,   477,   477,   477,   477,   477,   477,   477,   477,   476,   476,   476,   476,   476,   476,   476,   476,   479,   479,   479,   479,   479,   479,   479,   479,   478,   478,   478,   478,   478,   478,   478,   478,   473,   473,   473,   473,   473,   473,   473,   473,   472,   472,   472,   472,   472,   472,   472,   472,   475,   475,   475,   475,   475,   475,   475,   475,   474,   474,   474,   474,   474,   474,   474,   474
// CHECK: 494,   494,   494,   494,   494,   494,   494,   494,   495,   495,   495,   495,   495,   495,   495,   495,   492,   492,   492,   492,   492,   492,   492,   492,   493,   493,   493,   493,   493,   493,   493,   493,   490,   490,   490,   490,   490,   490,   490,   490,   491,   491,   491,   491,   491,   491,   491,   491,   488,   488,   488,   488,   488,   488,   488,   488,   489,   489,   489,   489,   489,   489,   489,   489,   511,   511,   511,   511,   511,   511,   511,   511,   510,   510,   510,   510,   510,   510,   510,   510,   509,   509,   509,   509,   509,   509,   509,   509,   508,   508,   508,   508,   508,   508,   508,   508,   507,   507,   507,   507,   507,   507,   507,   507,   506,   506,   506,   506,   506,   506,   506,   506,   505,   505,   505,   505,   505,   505,   505,   505,   504,   504,   504,   504,   504,   504,   504,   504
// CHECK: 520,   520,   520,   520,   520,   520,   520,   520,   521,   521,   521,   521,   521,   521,   521,   521,   522,   522,   522,   522,   522,   522,   522,   522,   523,   523,   523,   523,   523,   523,   523,   523,   524,   524,   524,   524,   524,   524,   524,   524,   525,   525,   525,   525,   525,   525,   525,   525,   526,   526,   526,   526,   526,   526,   526,   526,   527,   527,   527,   527,   527,   527,   527,   527,   537,   537,   537,   537,   537,   537,   537,   537,   536,   536,   536,   536,   536,   536,   536,   536,   539,   539,   539,   539,   539,   539,   539,   539,   538,   538,   538,   538,   538,   538,   538,   538,   541,   541,   541,   541,   541,   541,   541,   541,   540,   540,   540,   540,   540,   540,   540,   540,   543,   543,   543,   543,   543,   543,   543,   543,   542,   542,   542,   542,   542,   542,   542,   542
// CHECK: 554,   554,   554,   554,   554,   554,   554,   554,   555,   555,   555,   555,   555,   555,   555,   555,   552,   552,   552,   552,   552,   552,   552,   552,   553,   553,   553,   553,   553,   553,   553,   553,   558,   558,   558,   558,   558,   558,   558,   558,   559,   559,   559,   559,   559,   559,   559,   559,   556,   556,   556,   556,   556,   556,   556,   556,   557,   557,   557,   557,   557,   557,   557,   557,   571,   571,   571,   571,   571,   571,   571,   571,   570,   570,   570,   570,   570,   570,   570,   570,   569,   569,   569,   569,   569,   569,   569,   569,   568,   568,   568,   568,   568,   568,   568,   568,   575,   575,   575,   575,   575,   575,   575,   575,   574,   574,   574,   574,   574,   574,   574,   574,   573,   573,   573,   573,   573,   573,   573,   573,   572,   572,   572,   572,   572,   572,   572,   572
// CHECK: 588,   588,   588,   588,   588,   588,   588,   588,   589,   589,   589,   589,   589,   589,   589,   589,   590,   590,   590,   590,   590,   590,   590,   590,   591,   591,   591,   591,   591,   591,   591,   591,   584,   584,   584,   584,   584,   584,   584,   584,   585,   585,   585,   585,   585,   585,   585,   585,   586,   586,   586,   586,   586,   586,   586,   586,   587,   587,   587,   587,   587,   587,   587,   587,   605,   605,   605,   605,   605,   605,   605,   605,   604,   604,   604,   604,   604,   604,   604,   604,   607,   607,   607,   607,   607,   607,   607,   607,   606,   606,   606,   606,   606,   606,   606,   606,   601,   601,   601,   601,   601,   601,   601,   601,   600,   600,   600,   600,   600,   600,   600,   600,   603,   603,   603,   603,   603,   603,   603,   603,   602,   602,   602,   602,   602,   602,   602,   602
// CHECK: 622,   622,   622,   622,   622,   622,   622,   622,   623,   623,   623,   623,   623,   623,   623,   623,   620,   620,   620,   620,   620,   620,   620,   620,   621,   621,   621,   621,   621,   621,   621,   621,   618,   618,   618,   618,   618,   618,   618,   618,   619,   619,   619,   619,   619,   619,   619,   619,   616,   616,   616,   616,   616,   616,   616,   616,   617,   617,   617,   617,   617,   617,   617,   617,   639,   639,   639,   639,   639,   639,   639,   639,   638,   638,   638,   638,   638,   638,   638,   638,   637,   637,   637,   637,   637,   637,   637,   637,   636,   636,   636,   636,   636,   636,   636,   636,   635,   635,   635,   635,   635,   635,   635,   635,   634,   634,   634,   634,   634,   634,   634,   634,   633,   633,   633,   633,   633,   633,   633,   633,   632,   632,   632,   632,   632,   632,   632,   632
// CHECK: 648,   648,   648,   648,   648,   648,   648,   648,   649,   649,   649,   649,   649,   649,   649,   649,   650,   650,   650,   650,   650,   650,   650,   650,   651,   651,   651,   651,   651,   651,   651,   651,   652,   652,   652,   652,   652,   652,   652,   652,   653,   653,   653,   653,   653,   653,   653,   653,   654,   654,   654,   654,   654,   654,   654,   654,   655,   655,   655,   655,   655,   655,   655,   655,   665,   665,   665,   665,   665,   665,   665,   665,   664,   664,   664,   664,   664,   664,   664,   664,   667,   667,   667,   667,   667,   667,   667,   667,   666,   666,   666,   666,   666,   666,   666,   666,   669,   669,   669,   669,   669,   669,   669,   669,   668,   668,   668,   668,   668,   668,   668,   668,   671,   671,   671,   671,   671,   671,   671,   671,   670,   670,   670,   670,   670,   670,   670,   670
// CHECK: 682,   682,   682,   682,   682,   682,   682,   682,   683,   683,   683,   683,   683,   683,   683,   683,   680,   680,   680,   680,   680,   680,   680,   680,   681,   681,   681,   681,   681,   681,   681,   681,   686,   686,   686,   686,   686,   686,   686,   686,   687,   687,   687,   687,   687,   687,   687,   687,   684,   684,   684,   684,   684,   684,   684,   684,   685,   685,   685,   685,   685,   685,   685,   685,   699,   699,   699,   699,   699,   699,   699,   699,   698,   698,   698,   698,   698,   698,   698,   698,   697,   697,   697,   697,   697,   697,   697,   697,   696,   696,   696,   696,   696,   696,   696,   696,   703,   703,   703,   703,   703,   703,   703,   703,   702,   702,   702,   702,   702,   702,   702,   702,   701,   701,   701,   701,   701,   701,   701,   701,   700,   700,   700,   700,   700,   700,   700,   700
// CHECK: 716,   716,   716,   716,   716,   716,   716,   716,   717,   717,   717,   717,   717,   717,   717,   717,   718,   718,   718,   718,   718,   718,   718,   718,   719,   719,   719,   719,   719,   719,   719,   719,   712,   712,   712,   712,   712,   712,   712,   712,   713,   713,   713,   713,   713,   713,   713,   713,   714,   714,   714,   714,   714,   714,   714,   714,   715,   715,   715,   715,   715,   715,   715,   715,   733,   733,   733,   733,   733,   733,   733,   733,   732,   732,   732,   732,   732,   732,   732,   732,   735,   735,   735,   735,   735,   735,   735,   735,   734,   734,   734,   734,   734,   734,   734,   734,   729,   729,   729,   729,   729,   729,   729,   729,   728,   728,   728,   728,   728,   728,   728,   728,   731,   731,   731,   731,   731,   731,   731,   731,   730,   730,   730,   730,   730,   730,   730,   730
// CHECK: 750,   750,   750,   750,   750,   750,   750,   750,   751,   751,   751,   751,   751,   751,   751,   751,   748,   748,   748,   748,   748,   748,   748,   748,   749,   749,   749,   749,   749,   749,   749,   749,   746,   746,   746,   746,   746,   746,   746,   746,   747,   747,   747,   747,   747,   747,   747,   747,   744,   744,   744,   744,   744,   744,   744,   744,   745,   745,   745,   745,   745,   745,   745,   745,   767,   767,   767,   767,   767,   767,   767,   767,   766,   766,   766,   766,   766,   766,   766,   766,   765,   765,   765,   765,   765,   765,   765,   765,   764,   764,   764,   764,   764,   764,   764,   764,   763,   763,   763,   763,   763,   763,   763,   763,   762,   762,   762,   762,   762,   762,   762,   762,   761,   761,   761,   761,   761,   761,   761,   761,   760,   760,   760,   760,   760,   760,   760,   760
// CHECK: 776,   776,   776,   776,   776,   776,   776,   776,   777,   777,   777,   777,   777,   777,   777,   777,   778,   778,   778,   778,   778,   778,   778,   778,   779,   779,   779,   779,   779,   779,   779,   779,   780,   780,   780,   780,   780,   780,   780,   780,   781,   781,   781,   781,   781,   781,   781,   781,   782,   782,   782,   782,   782,   782,   782,   782,   783,   783,   783,   783,   783,   783,   783,   783,   793,   793,   793,   793,   793,   793,   793,   793,   792,   792,   792,   792,   792,   792,   792,   792,   795,   795,   795,   795,   795,   795,   795,   795,   794,   794,   794,   794,   794,   794,   794,   794,   797,   797,   797,   797,   797,   797,   797,   797,   796,   796,   796,   796,   796,   796,   796,   796,   799,   799,   799,   799,   799,   799,   799,   799,   798,   798,   798,   798,   798,   798,   798,   798
// CHECK: 810,   810,   810,   810,   810,   810,   810,   810,   811,   811,   811,   811,   811,   811,   811,   811,   808,   808,   808,   808,   808,   808,   808,   808,   809,   809,   809,   809,   809,   809,   809,   809,   814,   814,   814,   814,   814,   814,   814,   814,   815,   815,   815,   815,   815,   815,   815,   815,   812,   812,   812,   812,   812,   812,   812,   812,   813,   813,   813,   813,   813,   813,   813,   813,   827,   827,   827,   827,   827,   827,   827,   827,   826,   826,   826,   826,   826,   826,   826,   826,   825,   825,   825,   825,   825,   825,   825,   825,   824,   824,   824,   824,   824,   824,   824,   824,   831,   831,   831,   831,   831,   831,   831,   831,   830,   830,   830,   830,   830,   830,   830,   830,   829,   829,   829,   829,   829,   829,   829,   829,   828,   828,   828,   828,   828,   828,   828,   828
// CHECK: 844,   844,   844,   844,   844,   844,   844,   844,   845,   845,   845,   845,   845,   845,   845,   845,   846,   846,   846,   846,   846,   846,   846,   846,   847,   847,   847,   847,   847,   847,   847,   847,   840,   840,   840,   840,   840,   840,   840,   840,   841,   841,   841,   841,   841,   841,   841,   841,   842,   842,   842,   842,   842,   842,   842,   842,   843,   843,   843,   843,   843,   843,   843,   843,   861,   861,   861,   861,   861,   861,   861,   861,   860,   860,   860,   860,   860,   860,   860,   860,   863,   863,   863,   863,   863,   863,   863,   863,   862,   862,   862,   862,   862,   862,   862,   862,   857,   857,   857,   857,   857,   857,   857,   857,   856,   856,   856,   856,   856,   856,   856,   856,   859,   859,   859,   859,   859,   859,   859,   859,   858,   858,   858,   858,   858,   858,   858,   858
// CHECK: 878,   878,   878,   878,   878,   878,   878,   878,   879,   879,   879,   879,   879,   879,   879,   879,   876,   876,   876,   876,   876,   876,   876,   876,   877,   877,   877,   877,   877,   877,   877,   877,   874,   874,   874,   874,   874,   874,   874,   874,   875,   875,   875,   875,   875,   875,   875,   875,   872,   872,   872,   872,   872,   872,   872,   872,   873,   873,   873,   873,   873,   873,   873,   873,   895,   895,   895,   895,   895,   895,   895,   895,   894,   894,   894,   894,   894,   894,   894,   894,   893,   893,   893,   893,   893,   893,   893,   893,   892,   892,   892,   892,   892,   892,   892,   892,   891,   891,   891,   891,   891,   891,   891,   891,   890,   890,   890,   890,   890,   890,   890,   890,   889,   889,   889,   889,   889,   889,   889,   889,   888,   888,   888,   888,   888,   888,   888,   888
// CHECK: 904,   904,   904,   904,   904,   904,   904,   904,   905,   905,   905,   905,   905,   905,   905,   905,   906,   906,   906,   906,   906,   906,   906,   906,   907,   907,   907,   907,   907,   907,   907,   907,   908,   908,   908,   908,   908,   908,   908,   908,   909,   909,   909,   909,   909,   909,   909,   909,   910,   910,   910,   910,   910,   910,   910,   910,   911,   911,   911,   911,   911,   911,   911,   911,   921,   921,   921,   921,   921,   921,   921,   921,   920,   920,   920,   920,   920,   920,   920,   920,   923,   923,   923,   923,   923,   923,   923,   923,   922,   922,   922,   922,   922,   922,   922,   922,   925,   925,   925,   925,   925,   925,   925,   925,   924,   924,   924,   924,   924,   924,   924,   924,   927,   927,   927,   927,   927,   927,   927,   927,   926,   926,   926,   926,   926,   926,   926,   926
// CHECK: 938,   938,   938,   938,   938,   938,   938,   938,   939,   939,   939,   939,   939,   939,   939,   939,   936,   936,   936,   936,   936,   936,   936,   936,   937,   937,   937,   937,   937,   937,   937,   937,   942,   942,   942,   942,   942,   942,   942,   942,   943,   943,   943,   943,   943,   943,   943,   943,   940,   940,   940,   940,   940,   940,   940,   940,   941,   941,   941,   941,   941,   941,   941,   941,   955,   955,   955,   955,   955,   955,   955,   955,   954,   954,   954,   954,   954,   954,   954,   954,   953,   953,   953,   953,   953,   953,   953,   953,   952,   952,   952,   952,   952,   952,   952,   952,   959,   959,   959,   959,   959,   959,   959,   959,   958,   958,   958,   958,   958,   958,   958,   958,   957,   957,   957,   957,   957,   957,   957,   957,   956,   956,   956,   956,   956,   956,   956,   956
// CHECK: 972,   972,   972,   972,   972,   972,   972,   972,   973,   973,   973,   973,   973,   973,   973,   973,   974,   974,   974,   974,   974,   974,   974,   974,   975,   975,   975,   975,   975,   975,   975,   975,   968,   968,   968,   968,   968,   968,   968,   968,   969,   969,   969,   969,   969,   969,   969,   969,   970,   970,   970,   970,   970,   970,   970,   970,   971,   971,   971,   971,   971,   971,   971,   971,   989,   989,   989,   989,   989,   989,   989,   989,   988,   988,   988,   988,   988,   988,   988,   988,   991,   991,   991,   991,   991,   991,   991,   991,   990,   990,   990,   990,   990,   990,   990,   990,   985,   985,   985,   985,   985,   985,   985,   985,   984,   984,   984,   984,   984,   984,   984,   984,   987,   987,   987,   987,   987,   987,   987,   987,   986,   986,   986,   986,   986,   986,   986,   986
// CHECK: 1006,   1006,   1006,   1006,   1006,   1006,   1006,   1006,   1007,   1007,   1007,   1007,   1007,   1007,   1007,   1007,   1004,   1004,   1004,   1004,   1004,   1004,   1004,   1004,   1005,   1005,   1005,   1005,   1005,   1005,   1005,   1005,   1002,   1002,   1002,   1002,   1002,   1002,   1002,   1002,   1003,   1003,   1003,   1003,   1003,   1003,   1003,   1003,   1000,   1000,   1000,   1000,   1000,   1000,   1000,   1000,   1001,   1001,   1001,   1001,   1001,   1001,   1001,   1001,   1023,   1023,   1023,   1023,   1023,   1023,   1023,   1023,   1022,   1022,   1022,   1022,   1022,   1022,   1022,   1022,   1021,   1021,   1021,   1021,   1021,   1021,   1021,   1021,   1020,   1020,   1020,   1020,   1020,   1020,   1020,   1020,   1019,   1019,   1019,   1019,   1019,   1019,   1019,   1019,   1018,   1018,   1018,   1018,   1018,   1018,   1018,   1018,   1017,   1017,   1017,   1017,   1017,   1017,   1017,   1017,   1016,   1016,   1016,   1016,   1016,   1016,   1016,   1016

